#include "virge.h"

void print_regex_err(const char *regex, const char *regex_err_msg)
{
    syslog(LOG_INFO,"REGEX ERROR (regcomp): regex '%s' is invalid: '%s'",regex,regex_err_msg);
    fprintf(stdout, "REGEX ERROR (regcomp): regex '%s' is invalid: '%s'",regex,regex_err_msg);
}

/*
 * check_extension_regex:
 * 
 * Checks filename (attachment) against the list of attachements 'denied' in configuration file (V_EXT)
 * This assumed extensions are regular expressions, and does regexp matching
 * If regexCheckExtension=0, just try to compile all of the regexps
 */
int check_extension_regex(char *regexCheckExtension)
{
    int regexErr;
    int regexExecStatus;
    int i;

    regex_t preg;
    regmatch_t pmatch;
    int	ecount = vlist_count(EXTENSIONS);

    for(i=0;i<ecount;i++){
	const char *regex = vlist(EXTENSIONS,i);

	if(regexCheckExtension){
	    virge_debug(4, "[ check_extension_regex() ] comparing '%s' => '%s'", regex, regexCheckExtension);
	}

	/* Compile regular expression - it is case insensitive
	 * We really should compile each extension once,
	 * but since the average email message has a single attachment,
	 * that actually doesn't buy us much.
	 */
	if ((regexErr = regcomp(&preg, regex, REG_ICASE | REG_EXTENDED))){
	    /*
	     * If 0 is not returned - some error occured
	     * Expressions are dodgy - let's make sure some normal error messages are returned to the user
	     * Hope it helps ;)
	     */
	    switch (regexErr) {
	    case REG_BADRPT:
		print_regex_err(regex, ERR_REG_BADRPT);
		break;
				
	    case REG_BADBR:
		print_regex_err(regex, ERR_REG_BADBR);
		break;
				
	    case REG_EBRACE:
		print_regex_err(regex, ERR_REG_EBRACE);
		break;
				
	    case REG_EBRACK:
		print_regex_err(regex, ERR_REG_EBRACK);
		break;
				
	    case REG_ERANGE:
		print_regex_err(regex, ERR_REG_ERANGE);
		break;
				
	    case REG_ECTYPE:
		print_regex_err(regex, ERR_REG_ECTYPE);
		break;
				
	    case REG_ECOLLATE:
		print_regex_err(regex, ERR_REG_ECOLLATE);
		break;
				
	    case REG_EPAREN:
		print_regex_err(regex, ERR_REG_EPAREN);
		break;
				
	    case REG_ESUBREG:
		print_regex_err(regex, ERR_REG_ESUBREG);
		break;
				
#ifdef REG_EEND
	    case REG_EEND:
		print_regex_err(regex, ERR_REG_EEND);
		break;
#endif
	    case REG_EESCAPE:
		print_regex_err(regex, ERR_REG_EESCAPE);
		break;
				
	    case REG_BADPAT:
		print_regex_err(regex, ERR_REG_BADPAT);
		break;
				
#ifdef REG_ESIZE
	    case REG_ESIZE:
		print_regex_err(regex, ERR_REG_ESIZE);
		break;
#endif
	    case REG_ESPACE:
		print_regex_err(regex, ERR_REG_ESPACE);
		break;
		    
	    default:
		print_regex_err(regex, "Unknown error");
	    }
	}
	else {
	    /* 0 is returned for a match */
	    if(regexCheckExtension){
		if ((regexExecStatus = regexec(&preg, regexCheckExtension, 1, &pmatch, 0)) == 0)			{
		    virge_debug(1, "[ check_extension_regex() ] match '%s' => '%s'", regex, regexCheckExtension);
	
		    /* We have to free it before we return */
		    regfree(&preg);
		    return(1);
		}
	    }
	}
	
	/* Free preg */
	regfree(&preg);
    }
	
    virge_debug(1, "[ check_extension_regex() ] extension checking finished - no matches");
    return(0);
}

/*
  Checks attachments in MIME message - rfc2045_decode() 'loops' through all of them, and invokes this function for each
*/
void check_attachments(struct rfc2045 *p, struct rfc2045id *id, void *ptr)
{
    const char *content_type, *transfer_encoding, *charset;
    off_t start, end, body;
    const char *disposition;
    const char *disposition_name;
    const char *disposition_filename;
    const char *content_name;
    off_t nlines, nbodylines;

    rfc2045_mimeinfo(p, &content_type, &transfer_encoding, &charset);
    rfc2045_mimepos(p, &start, &end, &body, &nlines, &nbodylines);
    rfc2045_dispositioninfo(p, &disposition, &disposition_name, &disposition_filename);
    content_name=rfc2045_contentname(p);

    /*	printf("START: [%lu] - END: [%lu] - BODY: [%lu] - LINES: [%lu] - TYPE: [%s]\n", (unsigned long)start, (unsigned long)end, (unsigned long)body, (unsigned long)nlines, content_type); */

    /* I hope this is safe way of passing things - if very big filename is supplied, seems like rfc2045 ignores it (didn't check for sure yet :) */
    if ( (disposition_filename && *disposition_filename) && (needs_rewrite != 1) ) {
	    virge_debug(3, "[ check_attachments() ] check disposition_filename: '%s'", disposition_filename);
	    needs_rewrite = check_extension_regex((char *)disposition_filename);
    }
    else if ( (disposition_name && *disposition_name) && (needs_rewrite != 1) ) {
	    virge_debug(3, "[ check_attachments() ] check disposition_name: '%s'", disposition_name);
	    needs_rewrite = check_extension_regex((char *)disposition_name);
    } else if ( (content_name && *content_name) && (needs_rewrite != 1) ) {
	    virge_debug(3, "[ check_attachments() ] check content_name: '%s'", content_name);
	    needs_rewrite = check_extension_regex((char *)content_name);
    }

    virge_debug(1, "[ check_attachments() ] status of check_attachments (needs_rewrite) for content-type: [%s] is [%d]",
		content_type, needs_rewrite);
}

/*
 * rewrite_attachments:
 * Rewrite the message so it doesn't include the problem attachment.
 */

void rewrite_attachments(struct rfc2045 *p, struct rfc2045id *id, void *ptr)
{
    const char *content_type, *transfer_encoding, *charset;
    off_t start, end, body;
    const char *disposition;
    const char *disposition_name;
    const char *disposition_filename;
    const char *content_name;
    off_t nlines, nbodylines;

    char *readData;
    unsigned long cur;
    unsigned long sizeRead;
    unsigned long sizeWritten;
	
    int skipFile = 0;
    char boundaryString[V_BUFSIZE];

    rfc2045_mimeinfo(p, &content_type, &transfer_encoding, &charset);
    rfc2045_mimepos(p, &start, &end, &body, &nlines, &nbodylines);
    rfc2045_dispositioninfo(p, &disposition, &disposition_name, &disposition_filename);
    content_name=rfc2045_contentname(p);

    virge_debug(4, "[ rewrite_attachments() ] START: [%lu] - END: [%lu] - LINES: [%lu] - TYPE: [%s]",
		(unsigned long)start, (unsigned long)end, (unsigned long)nlines, content_type);

    /* We will skip completely the section '0' - it contains info about whole message */
    if (attachCounter != 0) {
	    /* If attachCounter is 1, this is first 'body' part in the msg (after headers) */
	    if (attachCounter == 1) {
		    /* Let's take the headers from the message */
		    /* We position to the beginning of the stream (message) */
		if (fseek(input_file, 0, SEEK_SET) == -1){
		    virge_debug(2, "[ rewrite_attachments() ] failed to positioned at the beginning of the stream using lseek()");
		}
		else {
		    virge_debug(3, "[ rewrite_attachments() ] positioned at the beginning of the stream (stdin) using lseek()");
		}
		
		    /*
		      We allocate 'start+1' for readData
		      Basically, what we do is:
		      - position to 0 (beginning of the stream)
		      - get the 'start' of 2nd MIME part (actually body message)
		      - assume that 0 -> start contains headers (yet to be proven 100% :)
		      - get that, and also get the 'boundary' by "rewinding" to the beginning of the previous line and grabbing that line (and cutting '\n')
		    */
		
		    /* Do I need that +1? */
		    if ((readData = calloc(1, start+1)) == NULL)
			{
			    virge_debug(2, "[ rewrite_attachments() ] failed to calloc() [%lu] bytes", (unsigned long)start+1);
			    /* FIXME - we should probably isolate the mail and scream! However, mail that makes this calloc() fails is probably 'naughty' one, so... do we care? :) */
			}
		    else
			{
			    virge_debug(3, "[ rewrite_attachments() ] calloc()ed [%lu] bytes", (unsigned long)start+1);
			}
		
		    /* Read the data (headers) into readData buffer */
		    if ((sizeRead = fread(readData, 1, start, input_file)) == -1) {
			    virge_debug(2, "[ rewrite_attachments() ] failed to read() [%lu] fread into readData", (unsigned long)start);
			}
		    else {
			    virge_debug(3, "[ rewrite_attachments() ] read() [%lu] bytes into readData", sizeRead);
			}

		    sizeWritten = fwrite(readData, 1, start, rewrite_file);
		    if (sizeWritten != start){
			virge_debug(2, "[ rewrite_attachments() ] failed to write() [%lu] bytes", start);
		    }
		    else {
			virge_debug(3, "[ rewrite_attachments() ] wrote() [%lu] bytes ", sizeWritten);
		    }
			
		    /* Free the allocated memory */
		    free(readData);
		    virge_debug(3, "[ rewrite_attachments() ] free()ed readData (hopefully ;)");
	
		    /* Now, we get the boundary - very stupid way, but might actualy work in 100% of cases - depends on librfc2045 */
		    /* Set 'cur' to start-2 (so that we skip '\n' of previous line) */
		    virge_debug(3, "[ rewrite_attachments() ] about to start 'rewinding' back in order to get boundary");
		    cur = (unsigned long)start-2;
		    do
			{
			    fseek(input_file, cur--, SEEK_SET);
			} while (fgetc(input_file) != '\n');

		    virge_debug(3, "[ rewrite_attachments() ] seems like we have found the beginning of boundary line");

		    /* Boundary is fixed size */
		    fgets(boundary, sizeof(boundary)-1, input_file);

		    /* Strip the newline */
		    if (strchr(boundary, '\n')){
			*strchr(boundary, '\n') = 0;
		    }

		    virge_debug(3, "[ rewrite_attachments() ] boundary is: [%s] (hope it's right :)", boundary);
		}
	
	    /* FIXME - damn, rescanning all over again just to find out why we are rewriting?! FIXME */
	    if (disposition_filename && *disposition_filename)
		{
		    if (check_extension_regex((char *)disposition_filename))
			{
			    skipFile = 1;
			    virge_debug(3, "[ rewrite_attachments() ] rewriting message (disposition_filename) because filename '%s' is not allowed", disposition_filename);
			}
		}

	    if (disposition_name && *disposition_name)
		{
		    if (check_extension_regex((char *)disposition_name))
			{
			    skipFile = 2;
			    virge_debug(3, "[ rewrite_attachments() ] rewriting message (disposition_name) because filename '%s' is not allowed", disposition_name);
			}
		}
	
	    if (content_name && *content_name)
		{
		    if (check_extension_regex((char *)content_name))
			{
			    skipFile = 3;
			    virge_debug(3, "[ rewrite_attachments() ] rewriting message (content_name) because filename '%s' is not allowed", content_name);
			}
		}

	    /* Do we 'skip' (not send to recipient) this part or not? */
	    if (skipFile < 1) {
		    /* Ok, include this section into the mail/message */
		    long readLines = (unsigned long) nlines;
		    static char readLine[V_BUFSIZE];

		    virge_debug(3, "[ rewrite_attachments() ] about to read [%lu] lines", readLines);

		    if ((fseek(input_file, start, SEEK_SET)) == -1) {
			    virge_debug(2, "[ rewrite_attachments() ] failed to position at the start of the MIME section (position: [%lu]) using fseek()", (unsigned long)start);
			}
		    else {
			    virge_debug(3, "[ rewrite_attachments() ] positioned at the start of the MIME section (position: [%lu]) using fseek()", (unsigned long)start);
			}

		    while(readLines >= 0) {
			    if (!fgets(readLine, V_BUFSIZE-1, input_file)) {
				    /* Phew... if there is not 'ending' boundary, really unexpected stuff happened - loooop... :((( */
				event_log(LOG_ERROR,"[ rewrite_attachments() ] failed to get a line using fgets() for Message ID: '%s'",
					  messageID);

				/* This happens when rfc2045 lib returns some wierd data (msg doesn't have ending boundary) */
				/* We'll just stop break here... */
				readLines = 0;
				break;
			    }
			    else {
				    /* If first line read from MIME section is empty
				     * - skip it (it wouldn't be RFC compliant if boundary is followed by empty line
				     */
				    fputs(readLine,rewrite_file);

				    readLines = readLines - 1;
				}
			}

		    /* Add boundary */
		    snprintf(boundaryString, sizeof(boundaryString)-1, "\n%s\n", boundary);
		    if( fputs(boundaryString,rewrite_file)==EOF){
			virge_debug(2, "[ rewrite_attachments() ] failed to write() (boundaryString)");
		    }
		    else {
			virge_debug(3, "[ rewrite_attachments() ] wrote (boundaryString)");
		    }
		}
	    else {
		switch (skipFile){
		    /* disposition_filename */
		case 1:
		    vlist_append(REMOVED,disposition_filename);
		    virge_debug(3, "[ rewrite_attachments() ] added '%s' [disposition_filename] to REMOVED array", disposition_filename);
		    break;
				
		case 2:
		    vlist_append(REMOVED,disposition_name);
		    virge_debug(3, "[ rewrite_attachments() ] added '%s' [disposition_name] to REMOVED array", vlist_last(REMOVED));
		    break;

		case 3:
		    vlist_append(REMOVED,content_name);
		    virge_debug(3, "[ rewrite_attachments() ] added '%s' [content_name] to REMOVED array", vlist_last(REMOVED));
		    break;
		}
	    }
	}

    /* attachCounter is just used to 'count' the number of MIME
     * sections in the message - rfc2045_decode() invokes check_attachment() multiple times
     */
    attachCounter++;
}

void notify_removed_attachments(FILE *f)
{
    int next = 0;
    int remPos;
    int count = vlist_count(REMOVED);

    if (count==0){
	virge_debug(2, "[ notify_removed_attachments() ] We have been invoked, but seems that no attachments were removed. A bug!?");
	return;
    }

    virge_debug(3, "[ notify_removed_attachments() ] Going to create string w/ list of removed attachments");

    for (remPos = 0 ; remPos < count ; remPos++) {
	virge_debug(3, "[ notify_removed_attachments() ] Processing : %d/%s",
		    remPos,vlist(REMOVED,remPos));

	fprintf(f,"%-10s : \"%s\"\n",
		next==0 ? "REMOVED" : "",
		vlist(REMOVED,remPos));
    }
}
