#include "virge.h"
#include <assert.h>

/* Config file things.
 * strings allocated in virge_config.c
 */

char *VIRGE_DIR_MODE = "00700";		/* default dir mode */
char *ISOLATED_FILE_MODE = "";

char *V_SCAN_SOPHIE_SOCKET = "";
char *V_SCAN_TROPHIE_SOCKET = "";
char *V_SCAN_CLAMAV_SOCKET = "";

char *V_DIR = "";
char *V_LOGDIR = "";
char *V_TEMPDIR = "";
char *V_ISODIR = "";
char *V_ETCDIR = "";
char *RUNAS_USER = "";
char *PROCMAIL = "";
char *PROCMAIL_ARGS = "-Y -f ${from} -a ${aparam} -d ${to}";
char *SENDMAIL = "";
char *SENDMAIL_ARGS = "-t";
char *ADMIN = "";
char *VIRGE_FROM = "";

char *LOG_VDEBUG = "";
char *LOG_VIRGE = "";
char *LOG_ERROR = "";
char *LOG_ACTION = "";
char *LOG_VIRUS = "";

char *V_CONFIG = "";
char *V_EXT = "";
char *V_USERS = "";

char *V_MAILBODY = "";
char *V_PROGNAME = "";

char *V_TEMPLATE_NOTIFY_SENDER_VIRUS = "";
char *V_TEMPLATE_NOTIFY_RCPT_VIRUS = "";
char *V_TEMPLATE_NOTIFY_SENDER_ATTACH = "";

/*
  Reads/loads extensions from V_EXT file (by default is /virge/etc/extensions) - attachments are later checked
  against those extensions (could be full filenames, if you wish).
			
  Small explanation:
			  
  If you have extension line like:
			  
  "\.exe$ 	# crap" (1 space, 1 tab and 1 '#' after the extension)
			  
  it will be stripped to "\.exe$"
*/
void load_extensions(void)
{
    FILE *fext;
    char extension[MAXPATHLEN];
	
    if(access(V_EXT,0)!=0){
	virge_debug(1," [ load_extension() ] extension file '%s' not found.",V_EXT);
	return;
    }

    fext = fopen(V_EXT, "r");
    if (!fext){
	    event_log(LOG_ERROR,"FAILURE: Could not read extensions configuration file '%s'", V_EXT);
	    return;
    }

    while (fgets(extension, sizeof(extension)-1, fext))	{
	char *cc;

	cc = strchr(extension,'#');	/* remove after # */
	if(cc) *cc = 0;

	cc = strchr(extension,'\n');
	if(cc) *cc = 0;			/* remove \n */

	cc = strchr(extension,' ');
	if(cc) *cc = 0;			/* remove \n */

	cc = strchr(extension,'\t');
	if(cc) *cc = 0;			/* remove \n */

	if(extension[0]==0) continue;	/* ignore blanks */
	
	vlist_append(EXTENSIONS,extension);
	virge_debug(3, "[ load_extensions() ] added extension '%s'", extension);
    }
    fclose(fext);

    virge_debug(1, "[ load_extensions() ] [%d] extensions loaded", vlist_count(EXTENSIONS));
}

/*
 *  Reads/loads list of users for which mails should be delivered immediately (no checks made)
 */
void load_allowusers(void)
{
    FILE *fusers;
    char buf[1024];
	
    fusers = fopen(V_USERS, "r");
    if (!fusers){
	event_log(LOG_ERROR,"FAILURE: Could not read 'allowed users' configuration file '%s'", V_USERS);
	return;
    }

    while (fgets(buf, sizeof(buf)-1, fusers))	{
	/* Remove the \n if present */
	char *cc = strchr(buf,'\n');
	if(cc) *cc = '\000';

	vlist_append(ALLOW_USERS,buf);
    }
    fclose(fusers);

    virge_debug(1, "[ load_allowusers() ] loaded [%d] users", vlist_count(ALLOW_USERS));
}

/* config_find(buf,name)
 * See if the named configuration variable occures in buf.
 * If it does, return a pointer to the argument.
 */
const char *config_find(const char *buf,const char *name)
{
    const char *cc;

    if (strncasecmp(buf,name,strlen(name))!=0) return 0; /* not present */
    cc = strchr(buf,':');
    assert(cc);				/* better have a :, as we validated it previously */
    cc++;				/* skip past : */
    while(*cc && isspace(*cc))
		cc++;				/* allow any number of spaces after : */

    return cc;
}

/* PROCESS_OPT:
 * See if the buffer contains value 'name'. If so, set it and continue
 */

#define PROCESS_OPT(name,var) \
{   const char *v = config_find(buf,name);\
	if(v){var = strdup(v);\
	virge_debug(2, "[ virge_config] option '%s' set to '%s'",name,var);\
	continue;}\
} 

/* PROCESS_IOPT:
 * Like above, but with integer options
 */

#define PROCESS_IOPT(name,var) \
{\
   const char *v = config_find(buf,name);\
   if(v){\
       var = atoi(v);\
       virge_debug(2, "[ virge_config] option '%s' set to '%d'",name,var);\
       continue;\
   }\
} 

 
/* PROCESS_DIR_OPT:
 * If the value being set does not begin with a "/", prepend V_DIR
 */
#define PROCESS_DIR_OPT(name,basedir,var)\
{\
    const char *v = config_find(buf,name);\
    if(v){\
        if(v[0]=='/'){\
             /* Absolute directory */\
             var = strdup(v);\
             continue;\
	}\
        if(strlen(basedir)==0){\
            virge_debug(1, "[ virge_config] option '%s' requires %s to be set",name,#basedir);\
            exit(2);\
        }\
        var = malloc(strlen(basedir)+2+strlen(v));\
        strcpy(var,basedir);\
        strcat(var,"/");\
        strcat(var,v);\
        virge_debug(2, "[ virge_config] option '%s' set to '%s'",name,var);\
        if(strlen(var)==0){\
            virge_debug(1, "[ virge_config] option '%s' set to invalid value",name);\
            exit(2);\
        }\
        continue;\
    }\
}
 
        


/* Load configuration from a file */
int load_config_file(const char *file)
{
    FILE *fconfig;
    int line=0;
    char buf[1024];
		
    virge_debug(1,"[ load_config_file(%s) ]",file);

    /* Open the file first */
    fconfig = fopen(file,"r");
    if(!fconfig){
	return 0;			/* failure */
    }
    while(!feof(fconfig)){
	char *cc;

	memset(buf,0,sizeof(buf));
	fgets(buf, sizeof(buf)-1, fconfig);

	line++;

	cc = strchr(buf,'\n');
	if(cc) *cc = 0;

	cc = strchr(buf,'#');
	if(cc) *cc = 0;

	if(!buf[0]) continue;		/*  no line */
	
	if(strlen(buf)<3 || strchr(buf,':')==0){
	    virge_debug(2, "[ config_set() ] line %d of config file is invalid: '%s'",line,buf);
	    syslog(LOG_INFO,"FAILURE: line %d of config file is invalid '%s'", line,buf);
	    fprintf(stderr, "ERROR: line %d of config file is invalid '%s'",line,buf);
	    exit(1);
	}

	PROCESS_OPT("admin:", ADMIN);
	PROCESS_OPT("isolated_mode:", ISOLATED_FILE_MODE);
	PROCESS_OPT("mail_body:", V_MAILBODY);
	PROCESS_OPT("procmail:", PROCMAIL);
	PROCESS_OPT("procmail_args:", PROCMAIL_ARGS);
	PROCESS_OPT("runas_user:",RUNAS_USER);
	PROCESS_OPT("scan_sophie_socket:", V_SCAN_SOPHIE_SOCKET);
	PROCESS_OPT("scan_trophie_socket:", V_SCAN_TROPHIE_SOCKET);
	PROCESS_OPT("scan_clamav_socket:", V_SCAN_CLAMAV_SOCKET);
	PROCESS_OPT("sendmail:", SENDMAIL);
	PROCESS_OPT("sendmail_args:", SENDMAIL_ARGS);
	PROCESS_OPT("virge_dir:",     V_DIR);
	PROCESS_OPT("virge_dir_mode:", VIRGE_DIR_MODE);
	PROCESS_OPT("virge_from:", VIRGE_FROM);

	PROCESS_DIR_OPT("temp_dir:",  V_DIR, V_TEMPDIR);
	PROCESS_DIR_OPT("etc_dir:",   V_DIR, V_ETCDIR);
	PROCESS_DIR_OPT("log_dir:",   V_DIR, V_LOGDIR);

	PROCESS_DIR_OPT("log_action:", V_LOGDIR, LOG_ACTION);
	PROCESS_DIR_OPT("log_debug:",  V_LOGDIR, LOG_VDEBUG);
	PROCESS_DIR_OPT("log_error:",  V_LOGDIR, LOG_ERROR);
	PROCESS_DIR_OPT("log_virge:",  V_LOGDIR, LOG_VIRGE);
	PROCESS_DIR_OPT("log_virus:",  V_LOGDIR, LOG_VIRUS);

	PROCESS_DIR_OPT("template_notify_sender_virus:", V_ETCDIR, V_TEMPLATE_NOTIFY_SENDER_VIRUS);
	PROCESS_DIR_OPT("template_notify_rcpt_virus:", V_ETCDIR, V_TEMPLATE_NOTIFY_RCPT_VIRUS);
	PROCESS_DIR_OPT("template_notify_sender_attachment:", V_ETCDIR, V_TEMPLATE_NOTIFY_SENDER_ATTACH);

	PROCESS_DIR_OPT("allowusers_file:", V_ETCDIR, V_USERS);
	PROCESS_DIR_OPT("extensions_file:", V_ETCDIR, V_EXT);

	PROCESS_DIR_OPT("isolate_dir:", V_DIR, V_ISODIR);

	PROCESS_IOPT("debug_output:", DEBUG_OUTPUT);

	if(DEBUG_LEVEL==0){		/* command-line debug level overrides */
	    PROCESS_IOPT("debug_level:", DEBUG_LEVEL);
	}
	PROCESS_IOPT("defer_loadavg:", V_DEFER_LOADAVG);
	PROCESS_IOPT("deliver_loadavg:", V_DELIVER_LOADAVG);
	PROCESS_IOPT("notify_rcpt_virus:", NOTIFY_RCPT_VIRUS);
	PROCESS_IOPT("notify_sender_attachment:", NOTIFY_SENDER_ATTACHMENT);
	PROCESS_IOPT("notify_sender_virus:", NOTIFY_SENDER_VIRUS);
	PROCESS_IOPT("notify_use_header_rcpt:", NOTIFY_USE_HEADER_RCPT);
	PROCESS_IOPT("save_isolated:", SAVE_ISOLATED);
	PROCESS_IOPT("scan_clam:", V_SCAN_CLAMAV);
	PROCESS_IOPT("scan_avp:", V_SCAN_AVP);
	PROCESS_IOPT("scan_sophie:", V_SCAN_SOPHIE);
	PROCESS_IOPT("scan_trophie:", V_SCAN_TROPHIE);
	PROCESS_IOPT("virge_timeout:", VIRGE_TIMEOUT);
	PROCESS_IOPT("defer_on_fail", V_DEFER_ON_FAIL);
	PROCESS_IOPT("smtp_mode:", SMTP_MODE);
    }

    return(1);			/* success */
}

void load_virge_config()
{
    char *buf;

    buf = strdup(V_CONFIG_PATH);

    while(buf[0]){
	char *next = strchr(buf,':');	/* see if there is another part */
	if(next){
	    *next = '\000';		/* terminate */
	    next++;
	}
	if(load_config_file(buf)){
	    return;			/* config file successfully loaded */
	}
	if(!next){
	    break;			/* no more entries */
	}
	buf = next;			/* go to next name */
    }
    fprintf(stderr, "ERROR: Configuration file not found in '%s' - aborting!\n", V_CONFIG_PATH);
    virge_debug(2, "[ load_virge_config() ] Configuration file not found in %s - aborting!", V_CONFIG_PATH);
    exit(1);
}

/*
  Check the validity of current (loaded) configuration
  The 'checks' are pretty simple - don't rely on it!
*/
void check_current_config(void)
{
    int virge_dir_mode = strtoul(VIRGE_DIR_MODE,NULL,8);

    if (CHECK_CONFIG_VERBOSE){
	fprintf(stdout, "-- CONFIGURATION CHECK INITIATED\n\n");
    }

    /* Sanity check some of the options */
    if((virge_dir_mode & 00700) != 00700){
	fprintf(stderr,"invalid virge_dir_mode.  Must be in the form 007xx.\n");
	fprintf(stderr,"your virge_dir_mode = %o\n",virge_dir_mode);
	exit(-1);
    }
    if (CHECK_CONFIG_VERBOSE){
	fprintf(stdout, "[   OK] VIRGE_DIR_MODE [%s] is valid!\n", VIRGE_DIR_MODE);
    }

    /* Does RUNAS_USER exist? */
    if (!getpwnam(RUNAS_USER))	{
	if (CHECK_CONFIG_VERBOSE){
	    fprintf(stdout, "[ERROR] RUNAS_USER [%s] does not exist!\n", RUNAS_USER);
	}

	virge_debug(2, "[ check_current_config() ] Invalid RUNAS_USER [%s]", RUNAS_USER);
	fprintf(stderr, "ERROR: RUNAS_USER [%s] does not exist in /etc/passwd - aborting!\n", RUNAS_USER);
	exit(1);
    }

    if (CHECK_CONFIG_VERBOSE){
	fprintf(stdout, "[   OK] RUNAS_USER [%s] exists!\n", RUNAS_USER);
    }

    /* Does PROCMAIL exist? */
    if (!file_exists(PROCMAIL)){
	if (CHECK_CONFIG_VERBOSE)
	    fprintf(stdout, "[ERROR] Procmail [%s] does not exist!\n", PROCMAIL);
		
	virge_debug(2, "[ check_current_config() ] Invalid PROCMAIL [%s]", PROCMAIL);
	fprintf(stderr, "ERROR: Invalid PROCMAIL settings [%s] - aborting!\n", PROCMAIL);
	exit(1);
    }
    if (CHECK_CONFIG_VERBOSE){
	fprintf(stdout, "[   OK] Procmail [%s] exists!\n", PROCMAIL);
    }

	/* Does SENDMAIL exist? */
    if (!file_exists(SENDMAIL)){
	if (CHECK_CONFIG_VERBOSE)
	    fprintf(stdout, "[ERROR] Sendmail [%s] does not exist!\n", SENDMAIL);

	virge_debug(2, "[ check_current_config() ] Invalid SENDMAIL [%s]", SENDMAIL);
	fprintf(stderr, "ERROR: Invalid SENDMAIL settings [%s] - aborting!\n", SENDMAIL);
	exit(1);

    }
    if (CHECK_CONFIG_VERBOSE){
	fprintf(stdout, "[   OK] Sendmail [%s] exists!\n", SENDMAIL);
    }


    /* Is ADMIN specified? */
    if (strlen(ADMIN) < 1){
	if (CHECK_CONFIG_VERBOSE)
	    fprintf(stdout, "[ERROR] ADMIN email is not specified\n");
		
	virge_debug(2, "[ check_current_config() ] ADMIN not specified");
	fprintf(stderr, "ERROR: ADMIN not specified - aborting\n");
	exit(1);
    }
    if (CHECK_CONFIG_VERBOSE) {
	fprintf(stdout, "[   OK] ADMIN email [%s] is specified!\n", ADMIN);
    }

    /* Is VIRGE_FROM specified? */
    if (strlen(VIRGE_FROM) < 1)	{
	if (CHECK_CONFIG_VERBOSE)
	    fprintf(stdout, "[ERROR] VIRGE_FROM email is not specified!\n");
		
	virge_debug(2, "[ check_current_config() ] VIRGE_FROM not specified");
	fprintf(stderr, "ERROR: VIRGE_FROM not specified - aborting!\n");
	exit(1);
    }
    if (CHECK_CONFIG_VERBOSE){
	fprintf(stdout, "[   OK] VIRGE_FROM email [%s] is specified!\n", VIRGE_FROM);
    }

    /* V_DIR check */
    /* We try to create a dir (if it doesn't exist) */
    demand_directory(V_DIR, "ROOT");
    if (CHECK_CONFIG_VERBOSE)
	{
	    if (!file_exists(V_DIR)){
		fprintf(stdout, "[ERROR] Main Virge directory [%s] does not exist!\n", V_DIR);
	    }
	    else{
		fprintf(stdout, "[   OK] Main Virge directory [%s] exists\n", V_DIR);
	    }
	}

    /* V_LOGDIR check */
    demand_directory(V_LOGDIR, "LOG");
    if (CHECK_CONFIG_VERBOSE) {
	if (!file_exists(V_LOGDIR)){
	    fprintf(stdout, "[ERROR] Virge log directory [%s] does not exist!\n", V_LOGDIR);
	}
	else {
	    fprintf(stdout, "[   OK] Virge log directory [%s] exists!\n", V_LOGDIR);
	}
    }

    /* Does LOG_VIRGE file exist? */
    if (CHECK_CONFIG_VERBOSE) {
	    if (!file_exists(LOG_VIRGE))
		{
		    fprintf(stdout, "[ERROR] LOG_VIRGE file [%s] does not exist!\n", LOG_VIRGE);
		    fprintf(stdout, "[ NOTE] LOG_VIRGE file [%s] can/will be created later!\n", LOG_VIRGE);
		}
	    else
		fprintf(stdout, "[   OK] LOG_VIRGE file [%s] exists!\n", LOG_VIRGE);
	}

    /* Does LOG_ERROR file exist? */
    if (CHECK_CONFIG_VERBOSE)	{
	    if (!file_exists(LOG_ERROR))
		{
		    fprintf(stdout, "[ERROR] LOG_ERROR file [%s] does not exist!\n", LOG_ERROR);
		    fprintf(stdout, "[ NOTE] LOG_ERROR file [%s] can/will be created later!\n", LOG_ERROR);
		}
	    else
		fprintf(stdout, "[   OK] LOG_ERROR file [%s] exists!\n", LOG_ERROR);
	}

    if (CHECK_CONFIG_VERBOSE)	{
	    /* Does LOG_ACTION file exist? */
	    if (!file_exists(LOG_ACTION))
		{
		    fprintf(stdout, "[ERROR] LOG_ACTION file [%s] does not exist!\n", LOG_ACTION);
		    fprintf(stdout, "[ NOTE] LOG_ACTION file [%s] can/will be created later!\n", LOG_ACTION);
		}
	    else
		fprintf(stdout, "[   OK] LOG_ACTION file [%s] exists!\n", LOG_ACTION);
	}

    if (CHECK_CONFIG_VERBOSE)	{
	    /* Does LOG_VIRUS file exist? */
	    if (!file_exists(LOG_VIRUS))
		{
		    fprintf(stdout, "[ERROR] LOG_VIRUS file [%s] does not exist!\n", LOG_VIRUS);
		    fprintf(stdout, "[ NOTE] LOG_VIRUS file [%s] can/will be created later!\n", LOG_VIRUS);
		}
	    else
		fprintf(stdout, "[   OK] LOG_VIRUS file [%s] exists!\n", LOG_VIRUS);
	}

    if (CHECK_CONFIG_VERBOSE) {
	/* Does LOG_VDEBUG file exist? */
	if (!file_exists(LOG_VDEBUG)) {
	    fprintf(stdout, "[ NOTE] LOG_VDEBUG file [%s] can/will be created later!\n", LOG_VDEBUG);
	}
	else {
	    fprintf(stdout, "[   OK] LOG_VDEBUG file [%s] exists!\n", LOG_VDEBUG);
	}
    }

    /* V_TEMPDIR check */
    demand_directory(V_TEMPDIR, "TEMP");
    if (CHECK_CONFIG_VERBOSE) {
	if (!file_exists(V_TEMPDIR))
	    fprintf(stdout, "[ERROR] Virge temp directory [%s] does not exist!\n", V_TEMPDIR);
	else
	    fprintf(stdout, "[   OK] Virge temp directory [%s] exists!\n", V_TEMPDIR);
    }

    /* V_ISODIR check */
    demand_directory(V_ISODIR, "ISOLATION");
    if (CHECK_CONFIG_VERBOSE)
	{
	    if (!file_exists(V_ISODIR))
		fprintf(stdout, "[ERROR] Virge isolation directory [%s] does not exist!\n", V_ISODIR);
	    else
		fprintf(stdout, "[   OK] Virge isolation directory [%s] exists!\n", V_ISODIR);
	}

    /* V_ETCDIR check */
    demand_directory(V_ETCDIR, "ETC");
    if (CHECK_CONFIG_VERBOSE)
	{
	    if (!file_exists(V_ETCDIR))
		fprintf(stdout, "[ERROR] Virge etc (config) directory [%s] does not exist!\n", V_ETCDIR);
	    else
		fprintf(stdout, "[   OK] Virge etc (config) directory [%s] exists!\n", V_ETCDIR);
	}

    /* Does V_EXT file exist? */
    if (CHECK_CONFIG_VERBOSE)
	{
	    if (!file_exists(V_EXT))
		{
		    fprintf(stdout, "[ERROR] V_EXT file [%s] does not exist!\n", V_EXT);
		    fprintf(stdout, "[ NOTE] V_EXT file [%s] can be created later!\n", V_EXT);
		}
	    else
		fprintf(stdout, "[   OK] V_EXT file [%s] exists!\n", V_EXT);
	}

    /* Does V_USERS file exist? */
    if (CHECK_CONFIG_VERBOSE)
	{
	    if (!file_exists(V_USERS))
		{
		    fprintf(stdout, "[ERROR] V_USERS file [%s] does not exist!\n", V_USERS);
		    fprintf(stdout, "[ NOTE] V_USERS file [%s] can be created later!\n", V_USERS);
		}
	    else
		fprintf(stdout, "[   OK] V_USERS file [%s] exists!\n", V_USERS);
	}

     /*
	    Mail (notification) templates
	*/
	 
	/* V_TEMPLATE_NOTIFY_SENDER_VIRUS */
    if (CHECK_CONFIG_VERBOSE)
	{
		if (!file_exists(V_TEMPLATE_NOTIFY_SENDER_VIRUS))
		    fprintf(stdout, "[ERROR] V_TEMPLATE_NOTIFY_SENDER_VIRUS file [%s] does not exist!\n", V_TEMPLATE_NOTIFY_SENDER_VIRUS);
		else
			fprintf(stdout, "[   OK] V_TEMPLATE_NOTIFY_SENDER_VIRUS file [%s] exists!\n", V_TEMPLATE_NOTIFY_SENDER_VIRUS);
	}
 
	/* V_TEMPLATE_NOTIFY_RCPT_VIRUS */
    if (CHECK_CONFIG_VERBOSE)
	{
		if (!file_exists(V_TEMPLATE_NOTIFY_RCPT_VIRUS))
		    fprintf(stdout, "[ERROR] V_TEMPLATE_NOTIFY_RCPT_VIRUS file [%s] does not exist!\n", V_TEMPLATE_NOTIFY_RCPT_VIRUS);
		else
			fprintf(stdout, "[   OK] V_TEMPLATE_NOTIFY_RCPT_VIRUS file [%s] exists!\n", V_TEMPLATE_NOTIFY_RCPT_VIRUS);
	}

	/* V_TEMPLATE_NOTIFY_SENDER_ATTACH */
    if (CHECK_CONFIG_VERBOSE)
	{
		if (!file_exists(V_TEMPLATE_NOTIFY_SENDER_ATTACH))
		    fprintf(stdout, "[ERROR] V_TEMPLATE_NOTIFY_SENDER_ATTACH file [%s] does not exist!\n", V_TEMPLATE_NOTIFY_SENDER_ATTACH);
		else
			fprintf(stdout, "[   OK] V_TEMPLATE_NOTIFY_SENDER_ATTACH file [%s] exists!\n", V_TEMPLATE_NOTIFY_SENDER_ATTACH);
	}

    /* NOTIFY_USE_HEADER_RCPT value */
    if (CHECK_CONFIG_VERBOSE)
	fprintf(stdout, "[   OK] NOTIFY_USE_HEADER_RCPT setting: [%d]\n", NOTIFY_USE_HEADER_RCPT);

	/* NOTIFY_SENDER_VIRUS value */
    if (CHECK_CONFIG_VERBOSE)
	fprintf(stdout, "[   OK] NOTIFY_SENDER_VIRUS setting: [%d]\n", NOTIFY_SENDER_VIRUS);

	/* NOTIFY_SENDER_ATTACHMENT value */
    if (CHECK_CONFIG_VERBOSE)
	fprintf(stdout, "[   OK] NOTIFY_SENDER_ATTACHMENT setting: [%d]\n", NOTIFY_SENDER_ATTACHMENT);

	/* NOTIFY_RCPT_VIRUS value */
    if (CHECK_CONFIG_VERBOSE)
	fprintf(stdout, "[   OK] NOTIFY_RCPT_VIRUS setting: [%d]\n", NOTIFY_RCPT_VIRUS);

	/* DIR_MODE value */
    if (CHECK_CONFIG_VERBOSE)
	fprintf(stdout, "[   OK] VIRGE_DIR_MODE setting: [%s]\n", VIRGE_DIR_MODE);

	/* ISOLATED_FILE_MODE value */
    if (CHECK_CONFIG_VERBOSE)
	fprintf(stdout, "[   OK] ISOLATED_FILE_MODE setting: [%s]\n", ISOLATED_FILE_MODE);

	/* V_SCAN_CLAMAV value */
    if (CHECK_CONFIG_VERBOSE)
	fprintf(stdout, "[   OK] V_SCAN_CLAMAV setting: [%d]\n", V_SCAN_CLAMAV);

	/* V_SCAN_CLAMAV_SOCKET value */
    if (CHECK_CONFIG_VERBOSE)
	fprintf(stdout, "[   OK] V_SCAN_CLAMAV_SOCKET: [%s]\n", V_SCAN_CLAMAV_SOCKET);

	/* V_SCAN_AVP value */
    if (CHECK_CONFIG_VERBOSE)
	fprintf(stdout, "[   OK] V_SCAN_AVP setting: [%d]\n", V_SCAN_AVP);

	/* V_SCAN_SOPHIE value */
    if (CHECK_CONFIG_VERBOSE)
	fprintf(stdout, "[   OK] V_SCAN_SOPHIE setting: [%d]\n", V_SCAN_SOPHIE);

	/* V_SCAN_SOPHIE_SOCKET value */
    if (CHECK_CONFIG_VERBOSE)
	fprintf(stdout, "[   OK] V_SCAN_SOPHIE_SOCKET: [%s]\n", V_SCAN_SOPHIE_SOCKET);

	/* V_SCAN_TROPHIE value */
    if (CHECK_CONFIG_VERBOSE)
	fprintf(stdout, "[   OK] V_SCAN_TROPHIE setting: [%d]\n", V_SCAN_TROPHIE);

	/* V_SCAN_TROPHIE_SOCKET value */
    if (CHECK_CONFIG_VERBOSE)
	fprintf(stdout, "[   OK] V_SCAN_TROPHIE_SOCKET: [%s]\n", V_SCAN_TROPHIE_SOCKET);

	/* SAVE_ISOLATED value */
    if (CHECK_CONFIG_VERBOSE)
	fprintf(stdout, "[   OK] SAVE_ISOLATED: [%d]\n", SAVE_ISOLATED);

	/* VIRGE_TIMEOUT value */
    if (CHECK_CONFIG_VERBOSE)
	fprintf(stdout, "[   OK] VIRGE_TIMEOUT: [%d]\n", VIRGE_TIMEOUT);

	/* V_MAILBODY */
    if (strlen(V_MAILBODY) < 1) {
	    if (CHECK_CONFIG_VERBOSE)
		fprintf(stdout, "[ERROR] V_MAILBODY is not set\n");
		
	    /* Set the default then */
	    V_MAILBODY = "mail.body";
	    virge_debug(1, "[ check_current_config() ] V_MAILBODY set to 'mail.body'");

	    if (CHECK_CONFIG_VERBOSE){
		fprintf(stdout, "[ NOTE] V_MAILBODY is now set to 'mail.body'\n");
	    }
	}
    else {
	    if (CHECK_CONFIG_VERBOSE)
		fprintf(stdout, "[   OK] V_MAILBODY is set to [%s]\n", V_MAILBODY);
	}

    /* V_DELIVER_LOADAVG value */
    if (CHECK_CONFIG_VERBOSE)
	fprintf(stdout, "[   OK] V_DELIVER_LOADAVG setting: [%d]\n", V_DELIVER_LOADAVG);

	/* DEBUG value */
    if (CHECK_CONFIG_VERBOSE){
	fprintf(stdout, "[ NOTE] DEBUG_OUTPUT set to [%d]\n", DEBUG_OUTPUT);
    }

    /* DEBUG_LEVEL value */
    if (CHECK_CONFIG_VERBOSE){
	fprintf(stdout, "[ NOTE] DEBUG_LEVEL set to [%d]\n", DEBUG_LEVEL);
    }



    if (CHECK_CONFIG_VERBOSE) {
	    fprintf(stdout, "\n-- CONFIGURATION CHECK COMPLETE\n");
	    exit(0);
	}

}

