#include "virge.h"
#include "notify.h"

/*
 * virge_openmail:
 * Opens an outgoing email connection.
 * Returns a FILE that you send the message to.
 * call virge_closemail(FILE) to send the message.
 *
 * Example:
 * FILE *f;
 * f = virge_openmail("erik@sendmail.org", "Instant message");
 * fprintf(f,"Hi, Erik. Thanks for writing sendmail.\n");
 * virge_closemail(f);
 * 
*/
FILE *virge_openmail(const char *to, const char *subject, const char *body)
{
    FILE *f;

    /* Make the arguments okay */
    if (subject == 0)
		subject = "";

    if(body == 0)
		body = "";

    virge_debug(1, "[ virge_openmail() ] mail sendmail mail : FROM [%s] TO: [%s] SUBJECT: [%s]", VIRGE_FROM, to, subject);

    /* In interactive mode, instead of calling sendmail, we call cat.
     * This will show the email that'd be delivered to the user in the terminal. */
    if (INTERACTIVE)
    {
		f = run_program(RUN_WRITE, "/bin/cat", "", 0);
    }
    else
    {
		/* Run the program used to deliver emails. */
		f = run_program(RUN_WRITE, SENDMAIL, SENDMAIL_ARGS,
			"from", RUN_STRING, VIRGE_FROM,
			"to", RUN_STRING, to,
			"subject", RUN_STRING, subject,
			0);
    }

    /* Check if call to run_program() succeeded. */
    if (!f)
		return(0);

    if(INTERACTIVE)
	{
		fprintf(f,"======================== RUNNING INTERACTIVE =======================\n");
		fprintf(f,"========= OUTGOING EMAIL DISPLAYED HERE INSTEAD OF BEING SEND ======\n");
    }

    fprintf(f,"To: %s\n", to);
    fprintf(f,"From: %s\n", VIRGE_FROM);

    if(subject[0])
		fprintf(f,"Subject: %s\n", subject);

    fprintf(f,"\n%s", body);

    return(f);
}

int virge_closemail(FILE *f)
{
    int r = 0, rtr;

    if(V_DELIVERY_SLEEP && !INTERACTIVE)
		sleep(V_DELIVERY_SLEEP);

    virge_debug(4, "[ virge_sendmail() ] sent.");

    if(INTERACTIVE)
		fprintf(f,"========================== END OF MESSAGE ============================\n");

    if (ferror(f))
    {
		event_log(LOG_ERROR, "Writing notification: %s", strerror(errno));
		r = -1;
    }

    fclose(f);

    /* We've to call run_terminate() before return()'ing, otherwise the process
     * running the MDA may become a zombie. */
    rtr = run_terminate();

    /* Check if something failed. */
    if (r)
	return(-1);

    /* Check if sendmail succeeded. */
    if (rtr)
    {
	event_log(LOG_ERROR, "sendmail failed");
	return(-1);
    }

    return(0);
}

int prepare_notification_mail_file(char *template_path, FILE *mh)
{
	FILE *th;
	int c;
	
	th = fopen(template_path, "r");
	if (!th)
		return(-1);
	
	
	while((c = fgetc(th)) > 0)
	{
		switch (c)
		{
			case '%':
			/* Get the character after the '%' sign */
			c = fgetc(th);
			
			/* Stop if we've encountered an error */
			if (c == EOF)
				break;

			/*
			   Check what is the character, and do the substitution
			   
			   %R = recipient
			   %D = date/time
			   %S = subject
			   %V = virus name
			   %A = list of removed attachments
			*/
			switch (c)
			{
				/* MAIL_FROM */
				case 'F':
				fprintf(mh, "%s", MAIL_FROM);
				break;
				
				/* RCPT_TO */
				case 'R':
				fprintf(mh, "%s", NOTIFY_USE_HEADER_RCPT ? NOTIFY_TO : RCPT_TO);
				break;

				/* Date/time */
				case 'D':
				fprintf(mh, "%s", NOTIFY_DATE);
				break;

				/* Subject */
				case 'S':
				fprintf(mh, "%s", NOTIFY_SUBJECT);
				break;

				/* VIRUS_NAME */
				case 'V':
				fprintf(mh, "%s", VIRUS_NAME);
				break;
				
				/* notify_removed_attachments() */
				case 'A':
				notify_removed_attachments(mh);
				break;				

				default:
				/* If we didn'ty recognize it, write %<character>, as it was originally */
				fprintf(mh, "%c%c", '%', c);
				break;
			}
			break;
			
			default:
			fprintf(mh, "%c", c);
			break;
		}
	}

	fclose(th);

	return(0);
}

int prepare_notification_mail_default(char *template_name, FILE *mh)
{
	int x = 0;
	int tpl_len = strlen(template_name);
	
	while (x < tpl_len)
	{	
		switch (template_name[x])
		{
			case '%':
			/* Get the character after the '%' sign */
			x++;
			
			/* Stop if we've encountered an error */
			if (x >= tpl_len)
				break;

			/*
			   Check what is the character, and do the substitution
			   
			   %R = recipient
			   %D = date/time
			   %S = subject
			   %V = virus name
			   %A = list of removed attachments
			*/
			switch (template_name[x])
			{
				/* MAIL_FROM */
				case 'F':
				fprintf(mh, "%s", MAIL_FROM);
				break;
				
				/* RCPT_TO */
				case 'R':
				fprintf(mh, "%s", NOTIFY_USE_HEADER_RCPT ? NOTIFY_TO : RCPT_TO);
				break;

				/* Date/time */
				case 'D':
				fprintf(mh, "%s", NOTIFY_DATE);
				break;

				/* Subject */
				case 'S':
				fprintf(mh, "%s", NOTIFY_SUBJECT);
				break;

				/* VIRUS_NAME */
				case 'V':
				fprintf(mh, "%s", VIRUS_NAME);
				break;
				
				/* notify_removed_attachments() */
				case 'A':
				notify_removed_attachments(mh);
				break;				

				default:
				/* If we didn'ty recognize it, write %<character>, as it was originally */
				fprintf(mh, "%c%c", '%', template_name[x]);
				break;
			}
			break;
			
			default:
			fprintf(mh, "%c", template_name[x]);
			break;
		}

		x++;
	}

	return(0);
}

int notify_sender(int notify_type)
{
    FILE *f;
    char *type = 0 ;

    switch(notify_type)
	{
    	case NOTIFY_VIRUS_FOUND:
		type = "VIRUS NOTIFICATION";
		break;
    
		case NOTIFY_BAD_ATTACHMENTS:
		type = "MAIL NOTIFICATION";
		break;
    }

    if (!(f = virge_openmail(MAIL_FROM,type,0)))
		return(-1);

    if (notify_type == NOTIFY_VIRUS_FOUND)
	{
		/* Read/parse the template, substitute any fields if needed. Return -1 if function fails */
		if ((prepare_notification_mail_file(V_TEMPLATE_NOTIFY_SENDER_VIRUS, f)) != 0)
		{
			if ((prepare_notification_mail_default(template_notify_sender_virus, f)) != 0)
				return(-1);
		}

		virge_debug(3, "[ notify_sender() ] Virus notification sent to sender [%s]", MAIL_FROM);
    }
    
	if (notify_type == NOTIFY_BAD_ATTACHMENTS)
	{
		/* Read/parse the template, substitute any fields if needed. Return -1 if function fails */
		if ((prepare_notification_mail_file(V_TEMPLATE_NOTIFY_SENDER_ATTACH, f)) != 0)
		{
			if ((prepare_notification_mail_default(template_notify_sender_attach, f)) != 0)
				return(-1);
		}

		virge_debug(3, "[ notify_sender() ] Attachment notification sent to sender [%s]", MAIL_FROM);
    }

    return(virge_closemail(f));
}

int notify_rcpt(int notify_type)
{
    FILE *f;

    if (notify_type == NOTIFY_VIRUS_FOUND)
	{
		if (!(f = virge_openmail(RCPT_TO,"VIRUS NOTIFICATION",0)))
		    return(-1);

		/* Read/parse the template, substitute any fields if needed. Return -1 if function fails */
		if ((prepare_notification_mail_file(V_TEMPLATE_NOTIFY_RCPT_VIRUS, f)) != 0)
		{
			if ((prepare_notification_mail_default(template_notify_rcpt_virus, f)) != 0)
				return(-1);
		}

		if (virge_closemail(f) == -1)
		    return(-1);
	
		virge_debug(3, "[ notify_rcpt() ] Virus notification sent to recipient [%s]", RCPT_TO);
    }

    return(0);
}


