#include "virge.h"
#include <assert.h>

void rfc2045_error(const char *errmsg)
{
    perror(errmsg);
    exit(0);
}

void do_print_structure(struct rfc2045 *p, struct rfc2045id *id, void *ptr)
{
    while (id)	{
	if (!id->next)	{
	    sections++;
	    virge_debug(4, "[ do_print_structure() ] one more section found - current count: [%d]", sections);
	}
	id=id->next;
    }
}

int decode_to_file(const char *p, size_t n, void *ptr)
{
    FILE	*fp=(FILE *)ptr;

    while (n)	{
	char ch = *p++;

	if (putc(ch, fp) == -1) {
	    perror("write");
	    return -1;
	}
	n--;
    }
    return (0);
}

/*
 * tempname:
 * Come up with a unique temporary name.
 * I'm not sure why we just can't call tempnam().
 */

char *tempname(const char *tempdir)
{
    char	hostnamebuf[256];
    static unsigned counter=0;
    char	*p = malloc(MAXPATHLEN+1);

    hostnamebuf[0] = 0;
    hostnamebuf[sizeof(hostnamebuf)-1]=0;
    gethostname(hostnamebuf, sizeof(hostnamebuf));
	
    snprintf(p, MAXPATHLEN, "%s/%lu.%u-%u.%s", tempdir, time(0), getpid(), counter++, hostnamebuf);
    return (p);
}

/* Print a section of the input file to output */
void do_print_section(struct rfc2045 *s, FILE *fp)
{
    off_t start, end, body;
    off_t nlines;
    off_t nbodylines;

    virge_debug(3, "[ do_print_section(%p,%p) ] ", s, fp);

    rfc2045_mimepos(s, &start, &end, &body, &nlines, &nbodylines);

    if (fseek(input_file, body, SEEK_SET) == -1) {
		perror("fseek");
		exit(1);
	}

	rfc2045_cdecode_start(s, &decode_to_file, fp);
	while (body < end)
	{
	char	buf[BUFSIZ];
	size_t	n=sizeof(buf);

		if ((off_t)n > end-body)	n=end-body;
		n=fread(buf, 1, n, input_file);
		if (n == 0) {
		    /* FIXME - For some reason, this get executed - check it later... */
		    perror("do_print_section:fread?");
		    return;
		}
		rfc2045_cdecode(s, buf, n);
		body += n;
	}
	rfc2045_cdecode_end(s);
}

void notfound(const char *p)
{
	fprintf(stderr, "reformime: MIME section %s not found.\n", p);
	exit(1);
}

char *get_suitable_filename(struct rfc2045 *r, const char *pfix, int ignore_filename)
{
    const char *disposition_s;
    const char *disposition_name_s;
    const char *disposition_filename_s;
    char	*filename_buf;
    const char *content_name_s;
    char	*p, *q;
    char	*dyn_disp_name=0;

    assert(pfix);

    rfc2045_dispositioninfo(r, &disposition_s, &disposition_name_s,
			    &disposition_filename_s);

	content_name_s=rfc2045_contentname(r);

	if (!disposition_filename_s || !*disposition_filename_s)
		disposition_filename_s=disposition_name_s;
	if (!disposition_filename_s || !*disposition_filename_s)
		disposition_filename_s=content_name_s;

	filename_buf=rfc2047_decode_simple(disposition_filename_s);

	if (!filename_buf)	{
		perror("rfc2047_decode_simple");
		exit(1);
	}

	if (strlen(filename_buf) > 32)	{
		p=filename_buf;
		q=filename_buf + strlen(filename_buf)-32;
		while ( (*p++ = *q++) != 0)
			;
	}

	disposition_filename_s=filename_buf;

	if (ignore_filename) {
	    char	numbuf[64];
	    static size_t counter=0;

	    sprintf(numbuf,"%d",++counter);

	    dyn_disp_name=malloc(strlen(disposition_filename_s) + strlen(numbuf)+2);
	    if (!dyn_disp_name) {
		perror("malloc");
		exit(1);
	    }
	    disposition_filename_s=strcat(strcat(strcpy( dyn_disp_name, numbuf), "-"),
					  disposition_filename_s);
	}
	else if (!disposition_filename_s || !*disposition_filename_s)	{
		dyn_disp_name=tempname(".");
		disposition_filename_s=dyn_disp_name+2;	/* Skip over ./ */
	}

	/* Create the filename with the prefix */
	p=malloc(strlen(pfix)+strlen(disposition_filename_s)+16);
	if (!p)	{
		perror("malloc");
		exit(1);
	}
	*p=0;
	strcpy(p, pfix);
	q=p+strlen(p);
	for (strcpy(q, disposition_filename_s); *q; q++){
	    if (!isalnum((int) *q) && *q != '.' && *q != '-'){
		*q='_';
	    }
	}

	if (dyn_disp_name) free(dyn_disp_name);
	free(filename_buf);
	virge_debug(3,"[ get_suitable_filename() ] returning '%s'",p);
	return (p);
}

void extract_file(struct rfc2045 *p, const char *filename, int argc, char **argv)
{
    char	*f;
    FILE	*fp;
    int	ignore=0;

    virge_debug(3,"[ extract_file(%p,%s,%d,...) ]",p,filename,argc);
    for (;;) {
	int	fd;

	f=get_suitable_filename(p, filename, ignore);
	if (!f)	return;

	fd=open(f, O_WRONLY|O_CREAT|O_EXCL, 0666);
	if (fd < 0) {
	    if (errno == EEXIST) {
// 		printf("%s exists.\n", f);
 	        event_log(LOG_ERROR,
		    "WARNING: tempfile %s already exists.",f);
		free(f);
		ignore=1;
		continue;
	    }

	    perror(f);
	    exit(1);
	}
	fp=fdopen(fd, "w");
	if (!fp) {
	    perror("fdopen");
	    exit(1);
	}
	break;
    }

    do_print_section(p, fp);
    if (fflush(fp) || ferror(fp)) {
	perror("write");
	exit(1);
    }
    fclose(fp);
    free(f);
}

void extract_section(struct rfc2045 *top_rfcp, const char *mimesection,
		     const char *extract_filename, int argc, char **argv,
		     void	(*extract_func)(struct rfc2045 *, const char *, int, char **))
{
    virge_debug(3,"[ extract_section(%p,%s,%s,%c,...,%p) ] ",top_rfcp,mimesection,extract_filename,argc,extract_func);
    if (mimesection) {
	top_rfcp=rfc2045_find(top_rfcp, mimesection);
	if (!mimesection)
	    notfound(mimesection);
	if (top_rfcp->firstpart) {
	    fprintf(stderr, "reformime: MIME section %s is a compound section.\n", mimesection);
	    exit(1);
	}
	(*extract_func)(top_rfcp, extract_filename, argc, argv);
	return;
    }

    /* Recursive */

    if (top_rfcp->firstpart) {
	for (top_rfcp=top_rfcp->firstpart; top_rfcp;
	     top_rfcp=top_rfcp->next)
	    extract_section(top_rfcp, mimesection,
			    extract_filename, argc, argv, extract_func);
	return;
    }

    if (!top_rfcp->isdummy){
	(*extract_func)(top_rfcp, extract_filename, argc, argv);
    }
}
