#include "virge.h"
#include <stdarg.h>
#include <assert.h>
#include <sys/wait.h>

typedef struct var 
{
    /* Names can't be bigger than 256 chars. See lookupVar(). */
    const char *name;
    unsigned int type;
    /* value will never be null in any instance of this struct. */
    const void *value;
    /* If type is not RUN_STRING, value will have to be converted to a string,
     * this is where the pointer to the string representation of this variable
     * should be placed. */
    char *str;
    struct var *next;
} VAR;

/**
 * List of variables set.
 */
static VAR *vars;

/**
 * PID of the process we're running.
 */
pid_t pid;

/**
 * Free VAR list.
 */
static void freeVarList(VAR *v)
{
    VAR *nv;

    for (; v; v = nv)
    {
	nv = v->next;
	if (v->str)
	    free(v->str);
	free(v);
    }
}

/**
 * Return value of variable named var.
 * Return variable value if found, NULL otherwise.
 */
static VAR *searchVar(const char *var)
{
    VAR *v;

    for (v = vars; v; v = v->next)
    {
	if (! strcmp(var, v->name))
	    return(v);
    }

    return(0);
}

/**
 * Get name of var, search for its value and return it.
 */
static VAR *lookupVar(const char **fmt)
{
    const char *s = (*fmt) + 1, *p;
    char vname[256];
    unsigned int l;
    VAR *v;

    /* Look for start of variable name. */
    if (**fmt != '{')
    {
	/* Missing the { char. */
	event_log(LOG_ERROR, "Format string missing {");
	return(0);
    }

    /* Look for end of variable name. */
    if (! (p = strchr(s, '}')))
    {
	/* Missing the } char. */
	event_log(LOG_ERROR, "Format string missing }");
	return(0);
    }

    /* Calculate size of variable name. */
    l = (unsigned int) p - (unsigned int) s;

    /* Copy variable name into vname so we can use it. */
    if (l >= sizeof(vname))
    {
	*fmt = p;
	event_log(LOG_ERROR, "Format string has a too big variable");
	return(0);
    }
    memcpy(vname, s, l);
    vname[l] = '\0';

    *fmt = ++ p;

    /* Search for variable value. */
    if (! (v = searchVar(vname)))
    {
	event_log(LOG_ERROR, "Variable \"%s\" doesn't exist", vname);
	return(0);
    }

    return(v);
}

/**
 * Create VAR's str entry and return it.
 * Type STRING does not require the str entry, so it isn't created.
 * Type VLIST returns NULL since it's special.
 */
static char *processVar(VAR *v)
{
    switch (v->type)
    {
	case RUN_STRING:
	    return((char *) v->value);

	case RUN_VLIST:
	    return(0);

	default:
	    event_log(LOG_ERROR, "FIXME: no var type %u", v->type);
	    return(0);
    }
}

/**
 * Unescape and replace ${var} by var's value.
 */
static int processArg(VLIST args, const char *fmt)
{
    const char *p;
    unsigned int i;
    char buf[8192], *bp = buf;
    unsigned int len = sizeof(buf);
    VAR *v;

    for (;;)
    {
	/* Check if current char is the escape char. */
	if (*fmt == '\\')
	{
	    fmt ++;
	    *(bp ++) = *(fmt ++);
	    len --;
	}
	/* Check if we've a variable. */
	else if (*fmt == '$')
	{
	    fmt ++;
	    /* Look up the variable's value. */
	    if (! (v = lookupVar(&fmt)))
		break;
	    /* Check if variable is of the special type VLIST. */
	    if (v->type == RUN_VLIST)
	    {
		/* Variables of type VLIST can't have anything before or after
		 * their specification in the format string. */
		if (bp != buf)
		{
		    /* Example: prefix${some_list_var}. */
		    event_log(LOG_ERROR, "Format has list variable prefixed");
		    return(-1);
		}
		else if (*fmt != '\0')
		{
		    /* Example: ${some_list_var}suffix */
		    event_log(LOG_ERROR, "Format has list variable suffixed");
		    return(-1);
		}
		/* Add all entries of this list into args. */
		for (i = 0; vlist((VLIST) v->value, i); i ++)
		    vlist_append(args, vlist((VLIST) v->value, i));
		/* VLIST variables are lonely, they don't have any other char
		 * with them, that's why we return here. */
		return(0);
	    }
	    else
	    {
		/* Get the string representation of variable v. */
		if (! (p = processVar(v)))
		    return(-1);
		/* Check if there's space left in buf for this string. */
		if ( (i = strlen(p)) >= len)
		    break;
		/* Copy variable's value to buf. */
		strcpy(bp, p);
		bp += i;
		len -= i;
	    }
	}
	else
	{
	    *(bp ++) = *(fmt ++);
	    len --;
	}

	/* Check if there's more data to parse. */
	if (! *fmt)
	{
	    *bp = '\0';
	    vlist_append(args, buf);
	    return(0);
	}

	/* Check if there's space left in buf for more data. */
	if (len == 0)
	    break;
    }

    event_log(LOG_ERROR, "processArg: buffer overflow");
    
    return(-1);
}

/**
 * This function facilitates the job of run_program.
 */
static FILE *run(int mode, const char *program, char *argv[], char *envp[])
{
    int pip[2], fd1, fd2, stdfd, null;
    FILE *fh;

    /* Create the pipe that'll be used to read or write data to the program we'll run. */
    if (pipe(pip) == -1)
    {
		event_log(LOG_ERROR, "pipe: %s", strerror(errno));
		return(0);
    }

    /* To make code cleaner. Instead of mode==RUN_?0:1, we'll just use fd1, fd2 and stdfd.
     * We'll assign to stdfd of the program we'll run fd2, and return fd1. */
    if (mode == RUN_READ)
    {
		fd1 = pip[0];
		fd2 = pip[1];
		stdfd = 1;
    }
    else
    {
		fd1 = pip[1];
		fd2 = pip[0];
		stdfd = 0;
    }

    /* Create FILE assigned to the side of the pipe the user wants. */
    if (! (fh = fdopen(fd1, mode == RUN_READ ? "r" : "w")))
    {
		event_log(LOG_ERROR, "fdopen: %s", strerror(errno));
		close(fd1);
		close(fd2);
		return(0);
    }

    /* Fork. We'll run the program on the new process (our child). */
    switch ( (pid = fork()))
    {
		case -1:
	    /* Failed fork()'ing. */
	    event_log(LOG_ERROR, "fork: %s", strerror(errno));
	    close(fd1);
	    close(fd2);
	    return(0);

		case 0: /* This is the child process. */
	    /* Close the side of the pipe we won't need. */
	    fclose(fh);

	    /* Open /dev/null, we'll need it below. */
	    if ( (null = open("/dev/null", O_RDWR)) == -1)
	    {
			event_log(LOG_ERROR, "Can't open /dev/null: %s", strerror(errno));
			exit(1);
	    }

	    /* Close std[in|out|err] so dup2 will use it below.
	     * dup2() is supposed to do this for us, but I don't know if this is the
	     * behaviour of dup2() on all platforms where Virge runs on. */
	    close(0);
	    close(1);
	    close(2);

	    /* Make /dev/null the stdin or stdout, and stderr. */
	    if (dup2(null, ! stdfd) == -1 || dup2(null, 2) == -1)
	    {
			event_log(LOG_ERROR, "dup2 /dev/null: %s", strerror(errno));
			exit(1);
	    }

	    /* Make pipe the std[in|out] of the new process. */
	    if (dup2(fd2, stdfd) == -1)
	    {
			event_log(LOG_ERROR, "dup2: %s", strerror(errno));
			exit(1);
	    }

	    /* Run program. */
	    execve(program, argv, envp);

	    /* If execve() returned, it failed. */
	    event_log(LOG_ERROR, "execve(%s): %s", program, strerror(errno));
	    exit(1);
    }

    /* Close the side of the pipe we won't need. */
    close(fd2);

    return(fh);
}

/**
 * Run program.
 * If you want to specify an argument that has a space, use single quote
 * around it. If your string has a quote, use a backslash to escape it.
 * Likewise, backslashes should be escaped by another backslash.
 * IMPORTANT: This function expects that VLIST be defined as char ***!!!
 */
FILE *run_program(int mode, const char *program, const char *argsstr, ...)
{
    /* This will be the enviroment passed to the program to be ran. */
    char *env[1] = {0};
    VLIST alist;
    const char *p, *e;
    unsigned int inQuote = 0, inStr = 0, atEnd = 0, l;
    char buf[8192];
    VAR *v;
    va_list ap;
    FILE *fh;

    /* Initialize variables. */
    alist = vlist_alloc();
    vlist_append(alist, program);

    /* Populate vars. */
    va_start(ap, argsstr);
    vars = 0;
    while ( (p = va_arg(ap, char *)))
    {
		assert ( (v = malloc(sizeof(*v))));
		v->name = p;
		v->type = va_arg(ap, unsigned int);
		v->value = va_arg(ap, void *);
		v->str = 0;

		v->next = vars;
		vars = v;
    }
    va_end(ap);

    /* Parse argsstr and populate alist. */
    for (p = argsstr, e = p; *p; e ++)
    {
		if (inStr)
		{
		    /* A backslash makes the following char have no special meaning. */
		    if (*e == '\\')
		    {
				if (e[1] != '\0')
				    e ++;
				continue;
		    }

		    /* We're inside a string. */
		    if (inQuote)
		    {
				/* We're inside a quoted string. */
				if (*e == '\'')
				{
				    atEnd = 1;
				}

				/* Check if we've reached the end of the parameters string.
				 * A quoted string should end in a quote char, otherwise, return error. */
				if (! *e)
				{
				    event_log(LOG_ERROR, "Missing quote in parameters string of \"%s\"",
					      program);
				    break;
				}
		    }
		    else
		    {
				/* We're inside an unquoted string. */
				if (*e == ' ' || ! *e)
				{
				    atEnd = 1;
				}
		    }

		    /* Check if we've reached the end of the string. */
		    if (atEnd)
		    {
				atEnd = 0;

				/* Calculate the size of the string. */
				l = (unsigned int) e - (unsigned int) p;

				/* Copy string to buf2, for processing by processArg. */
				if (l >= sizeof(buf))
				{
				    event_log(LOG_ERROR, "FIXME: buf2's too small");
				    break;
				}

				memcpy(buf, p, l);
				buf[l] = '\0';

				/* Unescape string and replace ${var} by var variable's value. */
				if (processArg(alist, buf) == -1)
				    break;

				p = e;

				inQuote = inStr = 0;
		    }
		}
		else
		{
		    /* We're looking for the start of a string. */
		    if (*e == '\'')
		    {
				/* The start of the string on a quoted string is the char after
				 * the quote char. */
				p = e + 1;
				inQuote = inStr = 1;
				continue;
		    }
		    else if (*e == '\0')
		    {
				p = e; /* *p == '\0' means we succeed. */
				break; /* Save some instructions and break here. */
		    }
		    else if (*e != ' ')
		    {
				p = e;
				inStr = 1;
		    }
		}
    }

    /* Check if parsing of the format string succeeded. */
    if (*p)
    {
		vlist_free(alist);
		freeVarList(vars);
		return(0);
    }

#ifdef DEBUG
    /* List all the arguments. */
    for (l = 0; vlist(alist, l); l ++)
	virge_debug(1, "%s Arg=%u Str=\"%s\"", program, l, vlist(alist, l));
#endif

    /* Run program. */
    fh = run(mode, program, *alist, env);

    /* Clean up. */
    vlist_free(alist);
    freeVarList(vars);

    return(fh);
}

/**
 * Wait for program ran by run_program() to exit and return its exit code.
 * The name of this function may be a little misleading. We'll not actually
 * terminate the program, we'll just wait for it to exit and then get its
 * exit code and return it.
 * Only if this function returns 0 you can be certain that the program ran
 * was successful, supposing that the program would return 0 on success,
 * any other return value indicates failure.
 */
int run_terminate()
{
    pid_t r;
    int s;

    /* Wait for process to terminate. */
    while ( (r = waitpid(pid, &s, 0)) != pid)
    {
	/* We check if waitpid failed here. */
	if (r == -1)
	{
	    /* Check if we've received a signal. */
	    if (errno == EINTR)
		continue;

	    /* If you see a ECHILD here, look at the manual for waitpid. */
	    event_log(LOG_ERROR, "waitpid: %s", strerror(errno));
	    return(-1);
	}
    }

    /* We found our process, let's check how it terminated. */
    if (WIFEXITED(s))
    {
	return(WEXITSTATUS(s));
    }
    else if (WIFSIGNALED(s))
    {
	event_log(LOG_ERROR, "program received signal (%u)", WTERMSIG(s));
	return(-1);
    }

    /* Your implementation of waitpid has a different behaviour than the one
     * documented on my boxes. */
    event_log(LOG_ERROR, "FIXME: unreachable area");
    return(-1);
}
