
#include "virge.h"

/*
  Scans a path with CLAMAV, and returns the result
  scanpath = path on *local* hard disk where attachments are unpacked
  
  Examples:
  
  scan_clam(dirspec);  scan recursively all files in the specified directory
  scan_clam(filespec); scan the specified file

        (C) 2003 J.J.G. steegers, All Rights Reserved.
  Parts (C) 2002 Simson L. Garfinkel, All Rights Reserved.

    VERSION 20030830  - hsx_AT_dds.nl
*/  
#define UNIX_MAX_PATH 108
#define MAX_SCAN_PATH 512
#define MAX_CLAM_RPLY 256
#define SCANCLAM_VERSION "0.3"

int clam_virscan (struct sockaddr_un clam_sock ,char const  *scanpath);
int pingpong_clam(struct sockaddr_un clam_sock);

/*
 * scan_clam(scanpath)
 * Scan a file or directory using clamd
 *
 */
int scan_clam(const char *scanpath) {
	char clamav_sockname[MAXPATHLEN]; 
	int exitcode;
	int vircount;
	struct sockaddr_un clam_socket_addr;

	virge_debug(2, 
	    "[ scan_clam() ] version %s - START",SCANCLAM_VERSION);
					    
	// prevent a buffer overflow
	if (strlen(scanpath) >= MAX_SCAN_PATH) {
		syslog(LOG_ERR,
		    "FAILURE: scanpath exceeds %d chars.",MAX_SCAN_PATH);
		return(-1);
	}
	/* clear and setup socket clam_sock	 */
	memset(&clam_socket_addr, 0, sizeof(clam_socket_addr));

	/* Check if socket name exists */
	if (V_SCAN_CLAMAV_SOCKET[0] == '\0') {
	    syslog(LOG_ERR,
		"FAILURE: clamd socket name not set");
	    virge_debug(2, 
		"[ scan_clam() ] FAILURE: V_SCAN_CLAMAV_SOCKET name not set");
	    return(-2);
	}

	if (strlen(&V_SCAN_CLAMAV_SOCKET[0]) > UNIX_MAX_PATH)
	{
	    syslog(LOG_ERR,
		"FAILURE: CLAMD socket name too long (>%d)",
		    UNIX_MAX_PATH);
	    virge_debug(2, 
		"[ scan_clam() ] FAILURE: V_SCAN_CLAMAV_SOCKET name too long");
	    return(-2);
	}	
	/* Copy the name into clamav_sockname */
	strncpy(clamav_sockname, 
		V_SCAN_CLAMAV_SOCKET, 
		sizeof(clamav_sockname)-1);
	virge_debug(2, 
	    "[ scan_clam() ] Socketname=%s",clamav_sockname);

	clam_socket_addr.sun_family=AF_UNIX;

	strncpy(clam_socket_addr.sun_path, 
		V_SCAN_CLAMAV_SOCKET, 
		sizeof(clam_socket_addr.sun_path) );

	/* 
	 *  verify if clamd is available 
	 */
    
	if (pingpong_clam(clam_socket_addr) != 0){
		virge_debug(2, 
		    "[ scan_clam() ] CLAMD is not responding to PINGPONG");
		syslog(LOG_ERR,
		    "FAILURE: PINGPONG Failed in scan_clam()");
		return(-1);
	}

	/* 
	 *  scan the files in scanpath 
	 */

	if ((vircount = clam_virscan(clam_socket_addr, scanpath)) < 0 ) {
		virge_debug(2, 
		    "[ scan_clam() ] clam_virscan failed._");
		syslog(LOG_ERR,
		    "FAILURE: scan_virscan()");
		return(-1);
	}

	if (vircount == 0) exitcode = 0; // None found
	else if (vircount == 1 ) exitcode = 1; // One found
	else if (vircount >= 0 ) exitcode = 2; // multiple found
	else exitcode = -1;	// error
     	virge_debug(3, 
	    "[ scan_clam() ] response received and processed [%d]", 
		    exitcode);
	virge_debug(2, 
	    "[ scan_clam() ] version %s - DONE",
		    SCANCLAM_VERSION);
	virge_debug(1, 
	    "[ scan_clam() ] Finishing: returns exitcode = %d",
		    exitcode);

	return(exitcode);
}



    /*
     * clam_virscan()
     * sends scan command to clamd over a UNIX socket
     * and receives the response.
     * RC = number of virii found.
     * RC < 0 on error, 
     */

int clam_virscan(struct sockaddr_un clam_sock , const char *scanpath) {

	int sd;
	int conn;
	int n , vircnt;
	FILE *fd;
	char clam_cmnd[MAX_SCAN_PATH];
	char clam_rply[MAX_CLAM_RPLY];
        char *name_ptr = NULL;

	/* connect for virus scan */
	/* open UNIX socket connection with CLAMD */
	if ( (sd = socket(AF_UNIX,SOCK_STREAM,0)) <0) {
		virge_debug(2, 
		    "[ scan_clam() ] failed to create UNIX socket.");
		syslog(LOG_ERR,
		    "FAILURE: Failed to create a socket in scan_clam()");
		return(-1);
	}

	conn = connect( sd, (struct sockaddr *) &clam_sock, 
			sizeof(struct sockaddr_un));

	if (conn == -1)	{
		virge_debug(2, 
		    "[ scan_clam() ] failed to connect() to CLAMD.");
		syslog(LOG_ERR,
		    "FAILURE: Failed to connect to CLAMD in scan_clam()");
		return(-1);
	}

	/* open file description for convenience */
        if((fd = fdopen(sd, "r")) == NULL) {
		syslog(LOG_ERR,
		    "FAILURE: Failed FDOPEN in scan_clam()");
	        return -1;
	}

        /* 
	 * "Craft" the CLAM query string... 
	 * It looks like: CONTSCAN /path.. 
	 * CONTSCAN continues scanning when finding a virus 
         */

	snprintf(clam_cmnd, MAX_SCAN_PATH , "CONTSCAN %s",  scanpath);
	virge_debug(3, 
	    "[ scan_clam() ] string to send to CLAM (%d bytes) = [%s]",
	    strlen(clam_cmnd), clam_cmnd);

	/* Send the request - read response */
 	if ((n = write(sd, clam_cmnd, strlen(clam_cmnd))) == -1)
	{
		virge_debug(2, 
		    "[ scan_clam() ] failed to write to CLAMD");
		syslog(LOG_ERR,
		    "FAILURE: Failed to write to CLAMD in scan_clam()");
		return(-1);
	}
	virge_debug(3, 
	    "[ scan_clam() ] string sent to CLAM = [%s]", clam_cmnd);
	
	/* Here comes the response... */
	virge_debug(3, "[ scan_clam() ] reading response from CLAMD");
	vircnt = 0;

	/* read replies until EOF and count the virusses found */
	while(fgets(clam_rply, MAX_CLAM_RPLY, fd)) 
	{ 
	    virge_debug(3, 
		"[ scan_clam() ] response=%s", clam_rply);
	    syslog(LOG_INFO,
		"scan_clam(): response=%s",clam_rply);

	    if(strstr(clam_rply, "FOUND\n")) { // virus found

	        (vircnt)++;

		if ( vircnt == 1 ) { // remember only the first

		    name_ptr = strrchr(clam_rply,'/');		   
		    name_ptr++;

		    virge_debug(3, 
			"[ scan_clam() ] virus: %s", name_ptr);
		    syslog(LOG_INFO,
			"scan_clam(): virus: %s",name_ptr);

		    if (strlen(name_ptr) > 7) {
		        name_ptr[strlen(name_ptr)-7]='\0';
		    } else { 
			name_ptr = "STRING SIZE ERROR in clam_scan()" ;
    			syslog(LOG_INFO,
			    "scan_clam(): ERROR in response string");
		    }

		    virge_debug(3, "[ scan_clam() ] v2=%s", name_ptr);
		    syslog(LOG_INFO,"scan_clam(): v2=%s",name_ptr);
		    VIRUS_NAME = strdup(name_ptr);

		} // end IF *vircnt=1

		virge_debug(2, 
		    "[ scan_clam() ] virus# %d found: %s", vircnt, name_ptr);

	    } //  end IF found

	} // end while

	virge_debug(3, 
	    "[ scan_clam() ] found %d vir[us|ii]", vircnt);

	if (fclose(fd)) { // Note: sd is also closed
		syslog(LOG_ERR,
		    "FAILED: Failed fclose() in scan_clam() errorno=%d",
			errno);
		return (-1);
	}
	return vircnt; // OK
}

/*
 * pingpong_clam(): send a PING and receive a PONG reply
 * to verify that the clamd daemon is alive and kicking..
 */
int pingpong_clam(struct sockaddr_un clam_sock)
{
    int sktd, conn, n;
    char tmpbuf[8];
	virge_debug(2, "[ pingpong_clam() ] START");

	/* open UNIX socket  */
	if ( (sktd = socket(AF_UNIX,SOCK_STREAM,0)) <0)
	{
		virge_debug(2, 
		    "[ scan_clam() ] failed to create UNIX socket.");
		syslog(LOG_ERR,
		    "FAILURE: Failed to create a socket in scan_clam()");
		return(-1);
	}

	conn = connect(	sktd, ( struct sockaddr *) &clam_sock, 
			sizeof(struct sockaddr_un));
	if (conn == -1) {
		virge_debug(2, 
		    "[ pingppong_clam() ] failed to connect to CLAMD.");
		syslog(LOG_ERR,
		    "FAILURE: Failed connect() in pingpong_clam()");
		return(-1);
	}

	/* send PING */
	sprintf(tmpbuf, "PING");
 	if ((write(sktd, tmpbuf, 4 )) == -1) {
		virge_debug(2, 
		    "[ pingpong_clam() ] failed write() to CLAMD");
		syslog(LOG_ERR,
		    "FAILURE: Failed write in pingpong_clam()");
		return(-1);
	}

	if ( ( n = read( sktd, tmpbuf, 5)) == -1 ) {
		virge_debug(2,
		     "[ pingpong_clam() ] failed read() from CLAMD");
		syslog(LOG_ERR,
		    "FAILURE: Failed read from CLAMD in pingpong_clam() ");
		return(-1);
	}
	// remove LF
	if (tmpbuf[4] == '\n') { 
	    tmpbuf[4] = '\0' ;
	} else {
		virge_debug(2,
		     "[ pingpong_clam() ] invalid PONG from CLAMD");
		syslog(LOG_ERR,
		    "FAILURE: invalid PONG from CLAMD in pingpong_clam()");
		return(-1);
	}

	virge_debug(3, 
	    "[ pingpong_clam() ] %d bytes with PONG-response=%s",
		n,tmpbuf);

	if (strstr(tmpbuf,"PONG"))	{
		virge_debug(2, 
		    "[ pingpong_clam() ] OK: PONG-reply from CLAMD");
	} else 	{
		virge_debug(2, 
		    "[ pingpong_clam() ] CLAMD failing PONG reply");
		syslog(LOG_ERR,
		    "FAILURE: invalid PONG reply in pingping_clam()");
		return(-1);
	}
	if  ( close(sktd) == 0){
		virge_debug(3, 
		    "[ pingpong_clam() ] skt close() OK");
	} else {
		virge_debug(2, 
		    "[ pingpong_clam() ] failing skt close()");
		syslog(LOG_ERR,
		    "FAILURE: CLAMD failed close() in clam_scan()");
		return(-1);    
	}
    return 0;
}

