/*
  Written by Andre Luiz dos Santos (andre at netvision.com.br)
*/

#include "virge.h"
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <assert.h>

#ifdef HAVE_SYSEXITS_H
#include <sysexits.h>
#endif

/*
 * When using mode1, we've to keep the connection to the SMTP server
 * opened between the calls to smtp_open_proxy() and smtp_deliver().
 * Here is where we'll keep this connection.
 */
static FILE *smtp = 0;

/*
 * When cleanup is called, we need to know if the client connection is
 * waiting for a reply or if it's received its reply already.
 * This var is set to 0 when nothing needs to be done, 1 when the client
 * is waiting for a reply.
 */
static int state = 0;

/*
 * Mode of operation.
 */
int SMTP_MODE = 1;

/*
 * Errors that readCmd() can return.
 */
#define S_UNKNOWN -2
#define S_ARGS -3

/*
 * Commands that readCmd() understands.
 * We won't speak ESMTP at this time.
 */
#define S_HELO 1
#define S_MAIL 2
#define S_RCPT 3
#define S_DATA 4
#define S_QUIT 5

/**
 * Read command from client, parse it and return its argument in var.
 * Return -1 on error, or one of the S_* defines above.
 */
static int readCmd(char *var, unsigned int len)
{
    int cmd;
    char buf[8192] = {0}, *arg, *p;

    /* Read command from client. */
    if (! fgets(buf, sizeof(buf), stdin))
    {
	event_log(LOG_ERROR, "readCmd: fgets from client: %s",
		  feof(stdin) ? "Connection lost" : strerror(errno));
	return(-1);
    }

    /* Check if command needed more than sizeof(buf) bytes. */
    if (buf[sizeof(buf) - 2] != '\0' && buf[sizeof(buf) - 2] != '\n')
    {
	/* The SMTP client sent us a huge line, only a huge email could
	 * possibly trigger this. */
	event_log(LOG_ERROR, "readCmd: buf's too small: increase it if you want this email");
	return(-1);
    }

    /* Parse command. */
#define CMD(s,n,c) if (! strncasecmp(buf, s, n)) { \
		cmd = c; arg = &buf[n]; }
    CMD("HELO", 4, S_HELO)
    else CMD("MAIL FROM:", 10, S_MAIL)
    else CMD("RCPT TO:", 8, S_RCPT)
    else CMD("DATA", 4, S_DATA)
    else CMD("QUIT", 4, S_QUIT)
    else
	/* Unknown command, we won't even bother with its arguments. */
	return(S_UNKNOWN);
#undef CMD

    /* Find command arguments. */
    for (; *arg && *arg == ' '; arg ++);

    /* Remove \r, \n and spaces from end of arguments. */
    for (p = &arg[strlen(arg) - 1]; p >= arg &&
	     (*p == ' ' || *p == '\r' || *p == '\n'); p --);
    *(++ p) = '\0';

    /* Parse arguments. */
    switch (cmd)
    {
	case S_HELO:
	    if (! *arg)
		return(S_ARGS);
	    break;

	case S_MAIL: case S_RCPT:
	    if (! *arg)
		return(S_ARGS);
	    /* If arg has < and >, put in var what's in the middle of them. */
	    if ( (p = strchr(arg, '<')))
	    {
		arg = ++ p;
		/* < has been found, > must be present. */
		if (! (p = strchr(p, '>')))
		    return(S_ARGS);
		*p = '\0';
	    }
	    /* Check if there're options after email address.
	     * Buggy client, we didn't advertise ESMTP. */
	    else if ( (p = strchr(arg, ' ')))
	    {
		/* Ignore any options. */
		*p = '\0';
	    }
	    break;
    }

    /* Check if we can copy the arguments to var. */
    if (strlen(arg) >= len)
    {
	/* smtp_open_proxy() uses a 8192 bytes long var, so this shouldn't
	 * happen, since the buf used to read commands is of the same size. */
	event_log(LOG_ERROR, "readCmd: var's too small: increase it if you want this email");
	return(S_ARGS);
    }

    /* Copy arguments to var. */
    strcpy(var, arg);

    return(cmd);
}

/**
 * Write reply to client.
 * Return 0 on success, -1 on error.
 */
static int writeReply(const char *rep, ...)
{
    char buf[8192] = {0};
    va_list ap;

    /* Format reply. */
    va_start(ap, rep);
    vsnprintf(buf, sizeof(buf), rep, ap);
    va_end(ap);

    /* Check if reply needed more than sizeof(buf) bytes. */
    if (buf[sizeof(buf) - 2] != '\0' && buf[sizeof(buf) - 2] != '\n')
    {
	/* This should never happen, since smtp_open_proxy() knows
	 * what it's doing, or at least it should. */
	event_log(LOG_ERROR, "writeReply: out of space in buf");
	return(-1);
    }

    /* Write reply to client. */
    if (fwrite(buf, strlen(buf), 1, stdout) != 1)
    {
	event_log(LOG_ERROR, "writeReply: fwrite to client: %s", strerror(errno));
	return(-1);
    }
    fflush(stdout);

    return(0);
}

/**
 * Receive email content.
 * input_file is where we should write the email.
 * Return 0 on success, -1 on error.
 */
int readEmailContent(FILE *input_file)
{
    char buf[8192], *p;
    int newLine = 1;

    /* Read email content line by line. */
    while (fgets(buf, sizeof(buf), stdin))
    {
	if (newLine)
	{
	    /* Check if line is escaped or if this is the last line. */
	    if(buf[0] == '.')
	    {
		if ((buf[1] == '\r' && buf[2] == '\n') ||
		    (buf[1] == '\n')){
		    /* End of email content. */
		    return(0);
		}
		/* Line is escaped. */
		p = &buf[1];
	    }
	    else
		p = buf;
	}

	/* Write line to file. */
	if (fwrite(buf, strlen(buf), 1, input_file) != 1)
	{
	    event_log(LOG_ERROR, "readEmailContent: fwrite to file: %s", strerror(errno));
	    return(-1);
	}

	/* Check if this is a full line/last part of a line. */
	if (strlen(buf) == sizeof(buf) - 1 && buf[sizeof(buf) - 2] != '\n')
	    newLine = 0;
	else
	    newLine = 1;
    }

    /* Check why fgets returned NULL. */
    event_log(LOG_ERROR, "readEmailContent: fgets from client: %s",
	      feof(stdin) ? "Connection lost" : strerror(errno));

    return(-1);
}

/**
 * Read reply from server.
 * re is the reply code expected.
 */
static int readReply(FILE *smtp, int re)
{
    char buf[8192], *p;
    int newLine = 1, rep = 0;

    /* Read command reply. */
    while (fgets(buf, sizeof(buf), smtp))
    {
	if (newLine)
	{
#define ISN(a) (a >= '0' && a <= '9')
	    /* Check if line is a valid reply line. */
	    if (buf[0] < '1' || buf[0] > '5' || ! ISN(buf[1]) || ! ISN(buf[2]) ||
		(buf[3] != ' ' && buf[3] != '-'))
	    {
		/* Remove trailing \r and \n. */
		for (p = &buf[strlen(buf) - 1]; p >= buf &&
			 (*p == '\r' || *p == '\n'); p --);
		*(++ p) = '\0';
		event_log(LOG_ERROR, "readReply: bad reply: [%s]", buf);
		return(-1);
	    }
#undef ISN
	    /* Check if line read is the last line of reply. */
	    if (buf[3] == ' ')
	    {
		/* Save the reply code in re. */
		rep = buf[0] - '0';
	    }
	}

	/* Check if this is a full line/last part of a line. */
	if (strlen(buf) == sizeof(buf) - 1 && buf[sizeof(buf) - 2] != '\n')
	    newLine = 0;
	else
	    newLine = 1;

	/* Check if line has been fully read and
	 * it's the last line of the reply. */
	if (newLine && rep)
	{
	    /* Check if reply was the expected. */
	    if (rep == re)
		return(0);
	    /* Remove trailing \r and \n. */
	    for (p = &buf[strlen(buf) - 1]; p >= buf &&
		     (*p == '\r' || *p == '\n'); p --);
	    *(++ p) = '\0';
	    event_log(LOG_ERROR, "readReply: negative reply: [%s]", buf);
	    return(-1);
	}
    }

    /* Check why fgets returned NULL. */
    event_log(LOG_ERROR, "readReply: fgets from server: %s",
	      feof(smtp) ? "Connection lost" : strerror(errno));

    return(-1);
}

/**
 * Write command to server.
 * Return 0 on success, -1 on error.
 */
static int writeCmd(FILE *smtp, const char *rep, ...)
{
    char buf[8192] = {0};
    va_list ap;

    /* Format command. */
    va_start(ap, rep);
    vsnprintf(buf, sizeof(buf), rep, ap);
    va_end(ap);

    /* Check if command needed more than sizeof(buf) bytes. */
    if (buf[sizeof(buf) - 2] != '\0' && buf[sizeof(buf) - 2] != '\n')
    {
	event_log(LOG_ERROR, "writeCmd: buf's too small: increase it if you want this email");
	return(-1);
    }

    /* Write command to server. */
    if (fwrite(buf, strlen(buf), 1, smtp) != 1)
    {
	event_log(LOG_ERROR, "writeCmd: fwrite to server: %s", strerror(errno));
	return(-1);
    }
    fflush(smtp);

    return(0);
}

/**
 * Write email content.
 */
int writeEmailContent(FILE *smtp, FILE *fh)
{
    char buf[8192];
    int newLine = 1, endInNewLine = 0;

    /* Read email content line by line. */
    while (fgets(buf, sizeof(buf), fh))
    {
	/* Check if line needs escaping. */
	if (newLine && buf[0] == '.')
	{
	    if (fputc('.', smtp) == EOF)
	    {
		event_log(LOG_ERROR, "writeEmailContent: fputc to server: %s", strerror(errno));
		return(-1);
	    }
	}

	/* Write line to server. */
	if (fwrite(buf, strlen(buf), 1, smtp) != 1)
	{
	    event_log(LOG_ERROR, "writeEmailContent: fwrite to server: %s", strerror(errno));
	    return(-1);
	}

	/* Check if this is a full line/last part of a line. */
	if (strlen(buf) == sizeof(buf) - 1 && buf[sizeof(buf) - 2] != '\n')
	    newLine = 0;
	else
	    newLine = 1;

	/* Check whether line ends in \n or not. */
	endInNewLine = buf[strlen(buf) - 1] == '\n' ? 1 : 0;
    }

    /* Check why fgets returned NULL. */
    if (! feof(fh))
    {
	event_log(LOG_ERROR, "writeEmailContent: fgets from file: %s", strerror(errno));
	return(-1);
    }

    /* Write terminator sequence. */
    if ( (! endInNewLine && fwrite("\r\n", 2, 1, smtp) != 1) ||
	 fwrite(".\r\n", 3, 1, smtp) != 1)
    {
	event_log(LOG_ERROR, "writeEmailContent: fwrite to server: %s", strerror(errno));
	return(-1);
    }

    return(0);
}

/**
 * Open connection to the SMTP server.
 * Return FILE on success, NULL on error.
 */
static FILE *smtpConnect()
{
    struct sockaddr_in p;
    int s;
    struct hostent *hp;
    unsigned int i;

    /* Resolve SMTP_HOST. */
    if (! (hp = gethostbyname(SMTP_HOST)))
    {
	event_log(LOG_ERROR, "smtp_deliver: couldn't resolve hostname: %s", SMTP_HOST);
	return(0);
    }

    /* Create TCP socket. */
    if ( (s = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP)) == -1)
    {
	event_log(LOG_ERROR, "smtp_deliver: socket: %s", strerror(errno));
	return(0);
    }

    /* Try to connect to one of the resolved addresses. */
    for (i = 0; hp->h_addr_list[i]; i ++)
    {
	p.sin_family = AF_INET;
	p.sin_addr.s_addr = *(unsigned long *) hp->h_addr_list[i];
	p.sin_port = htons(SMTP_PORT);

	/* Try to connect.
	 * On failure, try next address. */
	if (connect(s, (struct sockaddr *) &p, sizeof(p)) == -1)
	{
	    event_log(LOG_ERROR, "smtp_deliver: connect(%s:%d): %s",
		      inet_ntoa(p.sin_addr), SMTP_PORT, strerror(errno));
	    continue;
	}

	/* Associate socket to FILE structure.
	 * On failure, return error, we're most probably out of memory. */
	if (! (smtp = fdopen(s, "r+")))
	{
	    event_log(LOG_ERROR, "smtp_deliver: fdopen: %s", strerror(errno));
	    break;
	}

	/* Success. */
	return(smtp);
    }

    close(s);

    return(0);
}

/**
 * Read reply from SMTP server and send it to SMTP client.
 * Return reply code on success, -1 on error.
 */
static int smtpTransferServer2Client(FILE *smtp)
{
    char buf[8192] = {0}, *p;

    do
    {
	/* Read reply from server. */
	if (! fgets(buf, sizeof(buf), smtp))
	{
	    event_log(LOG_ERROR, "smtpTransferServer2Client: fgets from server: %s",
		      feof(smtp) ? "Connection lost" : strerror(errno));
	    return(-1);
	}

	/* Check if reply needed more bytes than sizeof(buf). */
	if (buf[sizeof(buf) - 2] != '\0' && buf[sizeof(buf) - 2] != '\n')
	{
	    event_log(LOG_ERROR, "smtpTransferServer2Client: reply's too big: increase buf");
	    return(-1);
	}

	/* Before sending the reply to the client, check if it's valid. */
#define ISN(a) (a >= '0' && a <= '9')
	/* Check if line is a valid reply line. */
	if (buf[0] < '1' || buf[0] > '5' || ! ISN(buf[1]) || ! ISN(buf[2]) ||
	    (buf[3] != ' ' && buf[3] != '-'))
	{
	    /* Remove trailing \r and \n. */
	    for (p = &buf[strlen(buf) - 1]; p >= buf &&
		     (*p == '\r' || *p == '\n'); p --);
	    *(++ p) = '\0';
	    event_log(LOG_ERROR, "smtpTransferServer2Client: bad reply: [%s]", buf);
	    return(-1);
	}
#undef ISN

	/* Write reply to client. */
	if (fwrite(buf, strlen(buf), 1, stdout) != 1)
	{
	    event_log(LOG_ERROR, "smtpTransferServer2Client: fwrite to client: %s",
		      strerror(errno));
	    return(-1);
	}
	fflush(stdout);

	/* Check if this was the last line of reply. */
    } while(buf[3] != ' ');

    return(buf[0] - '0');
}

/**
 * Read command from SMTP client and send it to SMTP server.
 * Return 0 on success, S_* vars for DATA and QUIT, -1 on error.
 */
static int smtpTransferClient2Server(FILE *smtp, char *var /* Must be 8192 bytes long. */)
{
    char buf[8192] = {0}, *p, *e;
    unsigned int cmd = 0;

    /* Read command from client. */
    if (! fgets(buf, sizeof(buf), stdin))
    {
	event_log(LOG_ERROR, "smtpTransferClient2Server: fgets from client: %s",
		  feof(stdin) ? "Connection lost" : strerror(errno));
	return(-1);
    }

    /* Check if command needed more bytes than sizeof(buf). */
    if (buf[sizeof(buf) - 2] != '\0' && buf[sizeof(buf) - 2] != '\n')
    {
	event_log(LOG_ERROR, "smtpTransferClient2Server: cmd's too big: increase buf");
	return(-1);
    }

    /* Find command.
     * Ignore tabs here too, because we never know what the server at
     * the other side will ignore, if it'll ignore anything at all. */
    for (p = buf; *p && (*p == ' ' || *p == '\t'); p ++);

    /* Write command to server. */
    if (fwrite(p, strlen(p), 1, smtp) != 1)
    {
	event_log(LOG_ERROR, "smtpTransferClient2Server: fwrite to server: %s",
		  strerror(errno));
	return(-1);
    }

    /* Check if command is DATA or QUIT. */
    if (p[4] == '\r' || p[4] == '\n' ||
	p[4] == ' ' || p[4] == '\t')
    {
	if (! strncasecmp(p, "DATA", 4))
	    return(S_DATA);
	else if (! strncasecmp(p, "QUIT", 4))
	    return(S_QUIT);
	else if (! strncasecmp(p, "MAIL", 4))
	    cmd = S_MAIL;
	else if (! strncasecmp(p, "RCPT", 4))
	    cmd = S_RCPT;

	if (cmd)
	{
	    /* The important thing here is: try to get the address exactly like the
	     * MTA will, and DO NOT do any checks on the syntax of the address, let
	     * this part of the job to the MTA, it'll decide what's valid and what's
	     * not and it'll tell us that by the its reply code. */

	    /* Look for start of address. */
	    if (! (p = strchr(p, ':')))
		return(0);
	    /* Check if address is surrounded by <>. */
	    if ( (e = strchr(++ p, '<')))
	    {
		/* Must have a matching >. */
		if (! (p = strchr(++ e, '>')))
		    /* Postfix will not accept this, note that other servers may
		     * accept this, probably with the leading < as part of the address. */
		    return(0);
		/* Postfix will ignore leading and trailing spaces and tabs on
		 * the address inside <>. */
		for (; *e && (*e == ' ' || *e == '\t'); e ++);
		for (p --; p >= e && (*p == ' ' || *p == '\t'); p --);
		*(++ p) = '\0';

		/* Copy address to var. */
		strcpy(var, e);
	    }
	    else
	    {
		/* Ignore spaces and tabs from start. */
		for (; *p && (*p == ' ' || *p == '\t'); p ++);
		/* Look for end of address. */
		for (e = p; *e && (*e == ' ' || *e == '\t' ||
				   *e == '\r' || *e == '\n'); e --);
		*(++ e) = '\0';

		/* Copy address to var. */
		strcpy(var, p);
	    }

	    return(cmd);
	}
    }

    return(0);
}

/**
 * Open connection to server SMTP and let the client in std[in|out] talk
 * to it directly, until the DATA command is seen.
 * When we see the DATA command, read the message from the client,
 * scan it, and if no virus is found, inject the message into the
 * server SMTP.
 *
 * Advantage: if sender or recipient envelope are rejected by SMTP server,
 *   we won't have to scan the message.
 * Disadvantage: while scanning an email, two SMTP processes are used.
 *
 * This function will only be called once during the life of this program.
 */
int smtp_open_proxy_mode1(FILE *input_file)
{
    int r;
    int lastCmd = 0;
    char buf[8192];

    /* Connect to the SMTP server. */
    if (! (smtp = smtpConnect()))
	return(-1);

    /* Do a bridge work, transfering data between the client SMTP and
     * the server SMTP where we should deliver the message. */
    for (;;)
    {
	/* Read reply from server and send it to client. */
	if ( (r = smtpTransferServer2Client(smtp)) == -1)
	    break;

	/* If command was QUIT, close connection. */
	if (lastCmd == S_QUIT)
	    break;
	/* If command was MAIL FROM and reply 2xx, set mail from. */
	else if (lastCmd == S_MAIL && r == 2)
	    assert(MAIL_FROM = strdup(buf));
	/* If command was RCPT TO and reply 2xx, add new recipient. */
	else if (lastCmd == S_RCPT && r == 2)
	{
	    vlist_append(RCPT_TOS, buf);
	}
	/* If we've seen the DATA command and the server replied
	 * with a reply code of 3xx, start getting the message content
	 * from the client. */
	else if (lastCmd == S_DATA && r == 3)
	{
	    if (readEmailContent(input_file) == -1)
		break;
	    /* Email content ready, scan it.
	     * The client is waiting for a reply, the server is waiting
	     * for the email content, smtp_deliver will do it later. */
	    state = 1;
	    return(0);
	}

	/* Read command from client and send it to server. */
	if ( (lastCmd = smtpTransferClient2Server(smtp, buf)) == -1)
	    break;
    }

    fclose(smtp);
    smtp = 0;

    event_log(LOG_ERROR, "No email received");

    return(-1);
}

/**
 * Receive email using the SMTP protocol. The client connection is
 * the std[in|out] FILEs.
 * input_file is where we should write the email.
 *
 * This function will only be called once during the life of this program.
 */
int smtp_open_proxy_mode2(FILE *input_file)
{
    char arg[8192];
    int cmd;
    unsigned int stage = 0;

    /* Write welcome message to client. */
    if (writeReply("220 virge.localhost SMTP Virge\r\n") == -1)
	return(-1);

#define REPLY(a...) { \
	if (writeReply(a) == -1) \
		return(-1); \
}
#define REPLYb(a...) { \
	REPLY(a); \
	break; \
}
    /* Read commands. */
    while ( (cmd = readCmd(arg, sizeof(arg))) != -1)
    {
	switch (cmd)
	{
	    case S_UNKNOWN:
		REPLYb("500 Unknown command\r\n");

	    case S_ARGS:
		REPLYb("501 Syntax error\r\n");

	    case S_HELO:
		/* TODO: Make the hostname configurable. */
		REPLYb("250 virge.localhost\r\n");

	    case S_MAIL:
		if (stage)
		    REPLYb("503 MAIL FROM already received\r\n");
		assert(MAIL_FROM = strdup(arg));
		stage ++;
		REPLYb("250 MAIL FROM accepted\r\n");

	    case S_RCPT:
		if (! stage)
		    REPLYb("503 MAIL FROM first please\r\n");
		vlist_append(RCPT_TOS, arg);
		stage ++;
		REPLYb("250 RCPT TO accepted\r\n");

	    case S_DATA:
		if (stage < 2)
		    REPLYb("503 MAIL FROM and RCPT TO first please\r\n");
		REPLY("354 Send the data\r\n");
		if (readEmailContent(input_file) == -1)
		    return(-1);
		/* The email data is ready to be scanned, as soon as the scan is
		 * finished, we'll send a positive reply to the client.
		 * smtp_deliver() is responsible for sending the reply. */
		state = 1;
		return(0);

	    case S_QUIT:
		/* If we see the QUIT command, it means that we didn't receive the
		 * DATA command, which in turn means there's no message to scan. */
		REPLY("221 Bye\r\n");
		event_log(LOG_ERROR, "No email received");
		return(-1);

	    default:
		event_log(LOG_ERROR, "FIXME: smtp_open_proxy_mode2, switch{default}");
		return(-1);
	}
    }
#undef REPLY
#undef REPLYb

    return(-1);
}

int smtp_open_proxy(FILE *input_file)
{
    switch (SMTP_MODE)
    {
	case 1:
	    return(smtp_open_proxy_mode1(input_file));

	case 2:
	    return(smtp_open_proxy_mode2(input_file));

	default:
	    event_log(LOG_ERROR, "FIXME: no smtp mode %d exists", SMTP_MODE);
	    return(-1);
    }
}

int smtp_deliver_mode1(const char *filename)
{
    FILE *fh;

    /* If anything goes wrong here, we don't want the client to receive
     * a reply of delivery succeed. */
    state = 0;

    for (;;)
    {
	/* Open file with email content. */
	if (! (fh = fopen(filename, "r")))
	{
	    event_log(LOG_ERROR, "smtp_deliver_mode1: fopen(%s): %s", filename, strerror(errno));
	    break;
	}

	/* Send email to SMTP server. */
	if (writeEmailContent(smtp, fh) == -1)
	{
	    fclose(fh);
	    break;
	}

	/* Email content already sent, close its file. */
	fclose(fh);

	/* We'll wait for the server to tell us if the email has been queued
	 * or rejected, then send a reply to the client. */
	if (readReply(smtp, 2) == -1)
	    break;

	/* Email queued by SMTP server, return positive reply to client. */
	if (writeReply("250 Delivered\r\n") == -1)
	    break;

	/* TODO: should I wait for commands from the client until it QUIT? */

	fclose(smtp);
	smtp = 0;

	return(0);
    }

    fclose(smtp);
    smtp = 0;

    return(-1);
}

int smtp_deliver_mode2(const char *filename)
{
    FILE *fh;
    unsigned int i;

    /* If anything goes wrong here, we don't want the client to receive
     * a reply of delivery succeed. */
    state = 0;

    /* Open file with email content. */
    if (! (fh = fopen(filename, "r")))
    {
	event_log(LOG_ERROR, "smtp_deliver_mode2: fopen(%s): %s", filename, strerror(errno));
	return(-1);
    }

    /* Connect to the SMTP server. */
    if (! (smtp = smtpConnect()))
    {
	fclose(fh);
	return(-1);
    }

#define REPLY(c) if (readReply(smtp, c) == -1) break;
#define CMD(c...) if (writeCmd(smtp, c) == -1) break;
    for(;;)
    {
	/* Wait for welcome message. */
	REPLY(2);

	/* TODO: make the hostname configurable, and maybe an option
	 * to not even send the HELO command. */
	CMD("HELO virge.localhost\r\n");
	REPLY(2);

	CMD("MAIL FROM: <%s>\r\n", MAIL_FROM);
	REPLY(2);

	for (i = 0; i < vlist_count(RCPT_TOS); i ++)
	{
	    CMD("RCPT TO: <%s>\r\n", vlist(RCPT_TOS, i));
	    REPLY(2);
	}
	/* Check if any of the recipients above failed. */
	if (i < vlist_count(RCPT_TOS))
	    break;

	CMD("DATA\r\n");
	REPLY(3);

	/* Send email to SMTP server. */
	if (writeEmailContent(smtp, fh) == -1)
	    break;
	REPLY(2);

	writeCmd(smtp, "QUIT\r\n");
	readReply(smtp, 2);

#undef CMD
#undef REPLY

	/* Reply to client that the email has been scanned and delivered. */
	writeReply("250 Delivered\r\n");

	/* TODO: should I wait for commands from the client until it QUIT? */

	fclose(fh);
	fclose(smtp);
	smtp = 0;

	return(0);
    }

    fclose(fh);
    fclose(smtp);
    smtp = 0;

    return(-1);
}

/**
 * The email has been virus scanned and it's clean, reinject the email
 * back into Postfix.
 *
 * This function will only be called once during the life of this program.
 */
int smtp_deliver(const char *filename)
{
    switch (SMTP_MODE)
    {
	case 1:
	    return(smtp_deliver_mode1(filename));

	case 2:
	    return(smtp_deliver_mode2(filename));

	default:
	    event_log(LOG_ERROR, "FIXME: no smtp mode %d exists", SMTP_MODE);
	    return(-1);
    }
}

/**
 * This function is called when SMTP proxy mode is enabled, every
 * time Virge is exiting.
 * Our responsibility is to tell the client when it's welcome to
 * come back with its email, or when it should go away and stop
 * bothering us.
 * There's no sense in checking for errors here, we're exiting anyway.
 *
 * This function will only be called once during the life of this program.
 */
void smtp_cleanup(int code)
{
    /* If we've a temporary error, just drop the connections we
     * may have open.
     * If client is waiting for a reply, send it a negative reply.
     * The client will retry the delivery in the next queue run. */
    if (code == EX_TEMPFAIL)
    {
	if (state == 1)
	{
	    writeReply("450 Virge failure\r\n");
	    /* Just to play safe, we set the state back to 0. */
	    state = 0;
	}
	return;
    }

    /* The client is waiting for a reply, we're exiting and we
     * don't want this email being sent again, so lie to the
     * client and make it think that the delivery succeeded. */
    if (state == 1)
    {
	writeReply("250 Trashed\r\n");
	/* Just to play safe, we set the state back to 0. */
	state = 0;
    }
}
