/* $Id: virge.c,v 1.4 2002/06/26 18:16:39 vanja Exp $ */

#include "virge.h"
#include <assert.h>

/****************************************************************
 ** VIRGE RUNTIME INFORMATION
 ****************************************************************/


int sections = 0;			/* Number of MIME sections in the message - needs to be global var */
int attachCounter = 0;			/* Number of attachments in the message */
long V_MAIL_SIZE = 0;			/* Mail size (in bytes, as result of read() in main()) */
int needs_rewrite = 0;			/* does message need rewriting? */
time_t VIRGE_START_TIME = 0;		/* Start time [result of time()] - just to be sure it's 0 at start ;) */

char *MAIL_FROM = "";
char *RCPT_TO = "";
VLIST RCPT_TOS;

/* 'header' that is attached to the message body when some attachments are removed */
char rewriteHeader[] = "\
Content-Type: text/plain; charset=\"us-ascii\"\n\
Content-Transfer-Encoding: 7bit\n\n\
\
+------------------------------------------------------------ --- -- -\n\
| NOTICE: One or more attachments have been removed from this mail\n\
|         Contact administrator if you require those attachments.\n\
+------------------------------------------------------------ --- -- -\n\
|\n\
| FILE(S) REMOVED:\n\
|\n";


/****************************************************************
 ** COMMAND LINE SWITCHES
 ****************************************************************/

int INTERACTIVE = 0;			/* are we interactive? set by isatty() */
int DEBUG_OUTPUT = 0;			/* 0=none; 1=stderr; 2=syslog; 3=both */
int CMDLINE_DEBUG = 0;			/* debug specified on command line; higher priority than config file */
int CHECK_REGEX = 0;			/* just check the reg expressions and exit. -R */
int CHECK_CONFIG_VERBOSE = 0;		/* Just check and print config file, then exit. -C */
char *arg_a = "";			/* default is '' (empty) */
int opt_keep = 0;

/* Virge lists that are dynamically loaded */
VLIST REMOVED;			// attachments that are removed
VLIST EXTENSIONS;		// extensions that are not allowed
VLIST ALLOW_USERS;		// users that are not scanned


/* Level:
 * 0 = none (then there is no point in setting DEBUG)
 * 1 = general info (start/end of functions)
 * 2 = errors (1 + error messages)
 * 3 = details (2 + details on certain operations that occur during virge operation)
 * 4 = phew (3 + big chunks of text (notification msg body, params to some functions, etc)
 */
int DEBUG_LEVEL = 0;

/*
 * Command line debug option(s) have higher priority than config ones - have to define it like this, since
 *  load_virge_config() is executed after cmdline params are checked
*/
int CMDLINE_DEBUG_LEVEL = 0;



/****************************************************************
 ** CONFIG OPTIONS.
 ** The following options are all set in the config file.
 ** see virge_config for details.
 ****************************************************************/



/****************************************************************
 ** Delivery modes.
 ** Only one of these should be set
 ****************************************************************/


int DELIVER_PROCMAIL = 1;		/* deliver with procmail; default */
int DELIVER_SMTP = 0;			/* Be a smtp proxy; requires SMTP_HOST and SMTP_PORT to be set */
int SMTP_PORT=25;			/* which port to use */
char *SMTP_HOST=0;			/* which host to use */


/****************************************************************
 ** Notify options
 ****************************************************************/

/* By default, don't use email address (recipient) from "To:" header
 * when sending notification to the virus/attachment sender
 */
int NOTIFY_USE_HEADER_RCPT = 0;

int NOTIFY_SENDER_VIRUS = 1;		/* Notify the sender when virus is found? */
int NOTIFY_SENDER_ATTACHMENT = 0;	/* Notify the sender when attachment is rejected? */
int NOTIFY_RCPT_VIRUS = 1;		/* Notify the recipient when virus is found? */
unsigned int VIRGE_TIMEOUT = 120;	/* Timeout value in seconds*/


int V_SCAN_AVP = 1;			/* Scan for viruses (AVP) by default */
int V_SCAN_CLAMAV = 1;			/* Scan for viruses (ClamAV) by default */
int V_SCAN_SOPHIE = 1;			/* Scan for viruses (Sophie) by default */
int V_SCAN_TROPHIE = 1;			/* Scan for viruses (Trophie) by default */

int SAVE_ISOLATED = 1;			/* If SAVE_ISOLATED == 0, infected files will NOT BE SAVED */

int V_DELIVER_LOADAVG = 4;		/* If loadavg reaches 4, just deliver mails */
int V_DEFER_LOADAVG = 4;		/* If loadavg reaches 4, DEFER */
int V_DEFER_ON_FAIL = 0;		/* normally, do not deliver the mail if we can't reach anti-virus */

int V_TEST=0;


char	boundary[V_BUFSIZE];


char	input_filename[MAXPATHLEN];	/* where we cache the input file */
char	rewrite_filename[MAXPATHLEN]; /* where the mail is rewritten */
FILE	*rewrite_file=0;			/* the temporary file */
FILE	*input_file;
char	scratchdir[MAXPATHLEN];

/* Information about the message */

char	headers[V_BUFSIZE];
char	*NOTIFY_SUBJECT="";	// message subject
char	*NOTIFY_DATE="";		// message date
char	*NOTIFY_TO="";		// message TO
char	*VIRUS_NAME="";
char	messageID[MAXPATHLEN];


#ifdef HAVE_SYSEXITS_H
#include <sysexits.h>
#endif

#ifdef HAVE_GETOPT_H
#include <getopt.h>
#endif

void virge_timeout(int sig)
{
    syslog(LOG_ERR, "[VIRGE TIMEOUT] ==> PID [%d], MESSAGE ID [%s], MAIL_FROM [%s], RCPT_TO [%s], TEMPFILE [%s]",
	   getpid(), messageID, MAIL_FROM, RCPT_TO, input_filename);
    exit(75);
}

void sig_exit(int sig)
{
    syslog(LOG_ERR, "OUCH! Caught signal '%d'. Mail from [%s] to [%s] IS NOT DELIVERED!!! "
	   "(probably is still hanging somewhere in temporary directory [%s]).",
	   sig, MAIL_FROM, RCPT_TO, V_TEMPDIR);
    exit(75);
}



/* Setup signal handlers.
 * Virge 2.0 set up signal handlers for ALL signals.
 * Turns out that you don't want to do that --- it hides bugs in the program,
 * which you really want to fix....
 */
void setup_signals()
{
    struct sigaction sa_exit;
    int sig_ret;

    /* Set signal handlers */
    sigemptyset(&sa_exit.sa_mask);
    sa_exit.sa_handler = sig_exit;

    if ((sig_ret = sigaction(SIGINT, &sa_exit, NULL)) < 0)
	syslog(LOG_INFO,"Error setting handler for SIGINT");

    if ((sig_ret = sigaction(SIGTERM, &sa_exit, NULL)) < 0)
	syslog(LOG_INFO,"Error setting handler for SIGTERM");

    if ((sig_ret = sigaction(SIGQUIT, &sa_exit, NULL)) < 0)
	syslog(LOG_INFO, "Error setting handler for SIGQUIT");

    /* If virge times out (hangs) for a while,
     * it needs to die and alert someone about the issue
     */
    sa_exit.sa_handler = virge_timeout;

    if ((sig_ret = sigaction(SIGALRM, &sa_exit, NULL)) < 0)
	syslog(LOG_INFO, "Error setting handler for SIGALRM");

    alarm(VIRGE_TIMEOUT);    /* Set the timeout */
}


/****************************************************************
 ** VIRGE EXIT ROUTINES.
 ** EVERYTHING ENDS UP HERE.
 ****************************************************************/

void virge_cleanup(struct rfc2045 *rfcbody,int code)
{
    /* smtp.c has its own special cleanup function. */
    if (DELIVER_SMTP)
    	smtp_cleanup(code);

    if(opt_keep==0){
	if(rfcbody){
	    rfc2045_free(rfcbody);
	}
	unlink(input_filename);
	if(scratchdir[0]){
	    remove_tempdir(scratchdir);
	    virge_debug(1, "[ virge_cleanup() ] scratchdir (%s) removed", scratchdir);
	}
    }
 
 	/* Remove the rewrite_filename */
	if (rewrite_filename[0] != '\0')
	{
		virge_debug(1, "[ virge_cleanup() ] Removing rewrite_filename (%s)", rewrite_filename);
	 	/* Will maybe add more checking later */
	 	unlink(rewrite_filename);
	}

    virge_debug(1, "--------------------------VIRGE exit(%d)----------------------------------\n",code);
    exit(code);
}

/* virge_defer:
 * Mail is not delivered.
 * PROCMAIL - return an error code.
 * SMTP     - terminate the socket
 */
void virge_defer(struct rfc2045 *rfcbody)
{
    virge_cleanup(rfcbody,EX_TEMPFAIL);
}

/* virge_deliver:
 * Mail is to be delivered. (Possibly a rewritten mail message)
 */
void virge_deliver(const char *filename,struct rfc2045 *rfcbody)
{
    int e = 0;

    fflush(input_file);			/* for good measure */

    if(DELIVER_SMTP){
	/* Note that with the current code, it's not necessary to check the return of
	 * smtp_deliver(), but in the future, it may be possible to receive an email
	 * through stdin and deliver it through SMTP. */
	if (smtp_deliver(filename) == -1)
	    e = EX_TEMPFAIL;
    }
    else if(DELIVER_PROCMAIL){
		e = procmail_deliver(filename, arg_a);
    }
    virge_cleanup(rfcbody, e);
}

/****************************************************************
 ** Usage
 ****************************************************************/

void usage(void)
{
    fprintf(stderr, "\n(Version: %s)\n\n", VIRGE_VERSION);
    fprintf(stderr, "Usage: %s [options] < [mailfile]\n", V_PROGNAME);
    fprintf(stderr, "These options are required for use as a procmail/sendmail filter:\n");
    fprintf(stderr, "  {-f|-s}    sender [supplied by sendmail]\n");
    fprintf(stderr, "  -d recepient    specifies who receives the mail\n");
    fprintf(stderr, "optional procmail args:\n");
    fprintf(stderr, "  -a <arg>     [supplied by sendmail; passed to procmail]\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "This is for operating virge as an SMTP proxy:\n");
    fprintf(stderr, "\n"); 
    fprintf(stderr,"   -Shost[:port]	connect to SMTP server at host:port\n");
    fprintf(stderr, "debugging options:\n");
    fprintf(stderr, "  -C   configuration check ONLY\n");
    fprintf(stderr, "  -R   extensions (regex) configuration check\n");
    fprintf(stderr, "  -DX  set DEBUG_LEVEL to X\n");
    fprintf(stderr, "  -OX  set DEBUG_OUTPUT output to (1=stderr, 2=logfile, 3=both)\n");
    fprintf(stderr, "  -h   help (this page)\n");
    fprintf(stderr, "  -t   test actions with results to stdout\n");
    fprintf(stderr, "  -k   Keep temp files (don't remove them)\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "------------------------------------------------------------------\n");
    fprintf(stderr, "Virge can be used as a sendmail/procmail filter, or as an SMTP proxy.\n");
    fprintf(stderr, "to test virge from the command line, use something like this:\n");
    fprintf(stderr, "  %s -t < MIME_message_body (without SMTP envelope) \n", V_PROGNAME);
    fprintf(stderr, "-------------------------------------------------------------------\n");
    exit(0);
}

/* virge_vscan()
 * 
 * The real heart of Virge --- scan all of the files in a directory to see if they contain
 * a virus.
 * 
 * Return 1 if infected, 0 if not.
 */

int virge_vscan(struct rfc2045 *rfcbody,const char *scratchdir)
{
    /* Now, we will scan for viruses. Check the V_SCAN_AVP setting first */
    if (V_SCAN_AVP)
	{
		int avp_scanret = scan_avp(scratchdir);

		switch(avp_scanret)
		{
			case 4:
		    virge_debug(1, "[ virge_vscan() ] AVP scan result: [%d] - virus [%s]", avp_scanret, VIRUS_NAME);
		    return 1;

			case 0:
		    virge_debug(1, "[ virge_vscan() ] AVP scan result: [0]");
		    break;

			default:
		    virge_debug(1, "[ virge_vscan() ] Unexpected response from AVP: [%d]", avp_scanret);
		    event_log(LOG_ACTION, "Mail isolated (unknown AVP response: [%d]) - "
			      "from=<%s>, to=<%s>, size=<%lu bytes>", avp_scanret, MAIL_FROM, RCPT_TO, V_MAIL_SIZE);
		    syslog(LOG_INFO,"NOTE: Mail has been isolated - unknown response from AVP received [%d]", avp_scanret);
		    isolate(0, "Unknown response from AVP received", input_filename, 1,
			    "Unknown response from AVP has been received. Mail isolated.");
		    break;
		}
    }
    else
	{
		virge_debug(1, "[ virge_vscan() ] AVP scan not performed (V_SCAN_AVP = 0)");
    }
			
    /* We will scan with Sophie only if AVP did not find virus already (so that we don't 'waste' resources) */
    if ( (V_SCAN_SOPHIE) )
	{
		int sophie_scanret = sophie_scandir(scratchdir);
		virge_debug(1, "[ virge_vscan() ] Sophie scan result: [%d]",
			sophie_scanret);

		if(sophie_scanret == 1)
		    return 1;	/* got a virus */

		/* Sophie error */
		if(sophie_scanret != 0)
		{	
			/*
			   If error string (in Sophie) are enabled, error msg will actually
			   be stored in VIRUS_NAME
			*/
			if (VIRUS_NAME[0] == '\0')
				VIRUS_NAME = strdup("unknown");

		    event_log(LOG_ERROR, "[ virge_vscan() ] sophie_scandir() returned %d (%s) for dir '%s'",
			      sophie_scanret,
				  VIRUS_NAME,
				  scratchdir);

		    if(V_DEFER_ON_FAIL)
			{
				/* do we defer on fail? */
				virge_defer(rfcbody);	/* yes, defer */
		    }
		}
    }
    else
	{
		virge_debug(1, "[ virge_vscan() ] SOPHIE scan not performed (V_SCAN_AVP = 0)");
    }
		
    /* Now scan with Trophie */
    if ( (V_SCAN_TROPHIE) )
	{
		int trophie_scanret = trophie_scandir(scratchdir);

		virge_debug(1, "[ virge_vscan() ] Trophie scan result: [%d]", trophie_scanret);

		if (trophie_scanret==1)
			return 1; /* got a virus */

		if(trophie_scanret!=0)
		{
			/* got a trophie error */
		    if(V_DEFER_ON_FAIL)
				virge_defer(rfcbody);
		}
    }  else {
	virge_debug(1, 
	    "[ virge_vscan() ] "
	    "TROPHIE scan not performed (V_SCAN_TROPHIE = 0)");
    }

    if (V_SCAN_CLAMAV) 	{
	int clam_scanret = scan_clam(scratchdir);

	switch(clam_scanret)	{ 

    	case 1:
	    virge_debug(1, 
	        "[ virge_vscan() ] CLAM scan result: [%d] - virus [%s]", 
		    clam_scanret, VIRUS_NAME);
	    return 1;

	case 2:
	    virge_debug(1, 
		"[ virge_vscan() ] CLAM scan result: "
		"[%d] - multiple virusses [%s]", 
		clam_scanret, VIRUS_NAME);
	    return 1;

	case 0:
	    virge_debug(1, "[ virge_vscan() ] CLAM scan result: [0]");
	    break;

	default:
	    virge_debug(1, 
		"[ virge_vscan() ] Unexpected response from CLAMAV: [%d]", 
		clam_scanret);
	    event_log(LOG_ACTION, 
		"Mail isolated (unknown CLAMAV response: [%d]) - "
		        "from=<%s>, to=<%s>, size=<%lu bytes>", 
			clam_scanret, MAIL_FROM, RCPT_TO, V_MAIL_SIZE);
	    syslog(LOG_INFO,"NOTE: Mail has been isolated - unknown response from CLAMAV received [%d]", clam_scanret);
	    isolate(0, "Unknown response from CLAMAV received", input_filename, 1,
		    "Unknown response from CLAMAV has been received. "
		    "Mail isolated.");
	    break;
		}
    }   else	{
	virge_debug(1, 
	    "[ virge_vscan() ] CLAMAV scan not performed (V_SCAN_CLAM = 0)");
    }
    
    return 0;
}


/* 
 * virge_checkrewrite()
 * 
 * Scan an email for illegal attachments. Returns 0 if message is fine,
 * 1 if message was rewritten and delivered.
 */
void virge_checkrewrite(struct rfc2045 *rfcbody)
{
    int removedPos;
    int rewrite_fd;

    /* If extensionCount is 0 = don't bother to decode/rewrite (everything is allowed) */
    if (vlist_count(EXTENSIONS) == 0)
		return;

    /* Check attachments, etc... */
    rfc2045_decode(rfcbody, &check_attachments, 0);

    if (needs_rewrite == 0)
	{
		/* Delivery of checked mail - no rewriting required, no viruses found */
		return;
    }
	
    /* At this point, we need to rewrite */
    snprintf(rewrite_filename, sizeof(rewrite_filename), "%s/%s", V_TEMPDIR, "rvirge-XXXXXX");

    /* Store the results into temporary file... */
    if ((rewrite_fd = mkstemp(rewrite_filename)) == -1) {
	virge_debug(2, "[ virge_checkrewrite() ] mkstemp(\"%s\") failed", rewrite_filename);
	virge_defer(rfcbody);
    }
    rewrite_file = fdopen(rewrite_fd,"w+");
    if(!rewrite_file){
	virge_debug(2,"[ virge_checkrewrite() ] fdopen failed");
	virge_defer(rfcbody);
    }

    virge_debug(1, "[ virge_checkrewrite() ] temporary filename for rewritten message created: '%s'", rewrite_filename);
    rfc2045_decode(rfcbody, &rewrite_attachments, 0);

    /* Now, we have modified the message, we need to add 'dislaimer' at the end
     * - it also helps us with the 'ending' boundary
     */
    fprintf(rewrite_file,"%s",rewriteHeader);

    for (removedPos = 0; removedPos < vlist_count(REMOVED); removedPos++)
	{
		/* Write the name of the stripped file */
		fprintf(rewrite_file,"| \"%s\"\n", vlist(REMOVED,removedPos));
		virge_debug(3, "[ virge_checkrewrite() ] added '%s' filename to the 'removed' footer in the message", vlist(REMOVED,removedPos));
    }

    /* More cosmetics... */
    fprintf(rewrite_file, "|\n");
    fprintf(rewrite_file, "+------------------------------------------------------------ --- -- -\n");

    /* And here comes the end boundary */
    fprintf(rewrite_file, "%s--\n",boundary);

    /* Close the temporary file, with new mail contents */
    fclose(rewrite_file);
    virge_debug(3, "[ virge_checkrewrite() ] rewrite_filename (%s) closed", rewrite_filename);
				
    /* Isolate the original mail (w/ attachment(s)) */
    virge_debug(1, "[ virge_checkrewrite() ] isolating mail (ID: '%s') because dodgy attachments found", messageID);
    if (isolate(1, "Questionable attachments found in the mail", input_filename, 1,
			"Mail has been isolated because potentially dangerous attachment(s) have been found.") == -1)
    {
		virge_defer(rfcbody);
    }

    /* Delivery of the new (rewritten) mail */
    virge_deliver(rewrite_filename, rfcbody);

}
	
void virge_drop_privs()
{
    struct passwd *userInfo;


    /* Get the uid/gid for RUN_AS user - for dropping privileges */
    if (!(userInfo = getpwnam(RUNAS_USER))) {
	virge_debug(2, "[ virge_drop_privs() ] failed to obtain UID/GID for user '%s'", RUNAS_USER);
	syslog(LOG_INFO,"FAILURE: Failed to obtain UID/GID for user '%s' - RUNNING AS ROOT", RUNAS_USER);
    } else {
	/* We have to chown() the debug logfile, since it gets opened as root,
	 * and when we drop privs... well, you guess ;)
	 */
	if (DEBUG_OUTPUT & 0x02) /* are we logging into file? */{
	    if (LOG_VDEBUG[0]) chown(LOG_VDEBUG, userInfo->pw_uid, userInfo->pw_gid);
	}
	
	/*
	 * Drop the privileges (hopefully ;)
	 * We don't need to run as root...
	 */
	virge_debug(1, "[ virge_drop_privs() ] running as user '%s'", RUNAS_USER);
	if ((setgid(userInfo->pw_gid)) == -1) {
	    virge_debug(2, "[ virge_drop_privs() ] failed to setgid() to [%d]", userInfo->pw_gid);
	    syslog(LOG_INFO,"FAILURE: Failed to setgid to '%d' - RUNNING AS ROOT", userInfo->pw_gid);
	} else {
	    virge_debug(3, "[ virge_drop_privs() ] managed to setgid() to [%d]", userInfo->pw_gid);
	}

	/* Init supplementary group access list */
	/* Don't exit if initgroups() failed */
	if (initgroups(RUNAS_USER, userInfo->pw_gid) == -1)
		virge_debug(1, "ERROR: initgroups() failed");

	if ((setuid(userInfo->pw_uid)) == -1) {
	    virge_debug(2, "[ virge_drop_privs() ] failed to setuid() to [%d]", userInfo->pw_uid);
	    event_log(LOG_ERROR,"FAILURE: Failed to setuid to '%d - RUNNING AS UID %d'",
		      userInfo->pw_uid,getuid());
	} else {
	    virge_debug(3, "[ virge_drop_privs() ] managed to setuid() to [%d]", userInfo->pw_uid);
	}
    }
    
}

void virge_create_messageid()
{
	unsigned long r;

	srandom(time(NULL) + getpid());
	r = random();

    snprintf(messageID, sizeof(messageID), "%.8lx-%.5d", r, getpid());
    virge_debug(1, "[ virge_create_messageid() ] Message ID created: [%s]", messageID);

}

void virge_read_message_stdin()
{
    char mailbuf[V_BUFSIZE];
    int bytes_read;
    int bytes_written;

    if(INTERACTIVE){
	fprintf(stderr,"Enter message:\n");
    }
    while( (bytes_read = read(0, mailbuf, sizeof(mailbuf))) > 0) {
	virge_debug(3, "[ virge_read_message_stdin() ] read [%d] bytes from stdin into mailbuf", bytes_read);
	
	bytes_written = fwrite(mailbuf,1,bytes_read,input_file);
	if(bytes_written != bytes_read){
	    virge_debug(2, "[ virge_read_message_stdin() ] failed to write() [%d] bytes into fd", bytes_read);
	    syslog(LOG_INFO,"FAILURE: Failed to write() [%d] bytes into fd", bytes_read);
	}
    }
}

/**
 * Make RCPT_TO.
 */
void make_rcpt_to()
{
    unsigned int i, j = 1 /* Count the terminator NULL here. */;
    char *p;

    for (i = 0; i < vlist_count(RCPT_TOS); i ++)
    {
	if (i)
	    j ++;
	j += strlen(vlist(RCPT_TOS, i));
    }

    assert(p = RCPT_TO = malloc(j));

    if (vlist_count(RCPT_TOS) == 0)
    {
	*p = '\0';
	return;
    }

    for (i = 0; i < vlist_count(RCPT_TOS); i ++)
    {
	if (i)
	    *(p ++) = ' ';

	strcpy(p, vlist(RCPT_TOS, i));
	p += strlen(p);
    }
}

/**
 * Make a new list of recipients in RCPT_TO, with all the addresses that are
 * not in ALLOW_USERS removed.
 * Those addresses are the ones that want to receive emails infected.
 */
void keep_only_allowed_rcpt_tos()
{
    VLIST l;
    unsigned int i, j;

    /* Make new list of allowed recipients. */
    l = vlist_alloc();
    for (i = 0; i < vlist_count(RCPT_TOS); i ++)
    {
	for (j = 0; j < vlist_count(ALLOW_USERS); j ++)
	{
	    if (! strcmp(vlist(ALLOW_USERS, j), vlist(RCPT_TOS, i)))
	    {
		/* Recipient is in ALLOW_USERS, keep it in. */
		vlist_append(l, vlist(RCPT_TOS, i));
		break;
	    }
	}
    }

    /* Free RCPT_TOS and RCPT_TO. */
    vlist_free(RCPT_TOS);
    free(RCPT_TO);

    /* Assign the new values to RCPT_TOS and RCPT_TO. */
    RCPT_TOS = l;
    make_rcpt_to();
}



int main(int argc, char *argv[])
{
    char *cc;
    char *message_buf;
    int	tempfile_fd;
    char scratchdir_prefix[MAXPATHLEN];
    char *cmdargs;			// remember the command line arguments
    int cmdsize=0;

    static int ISOLATE = 0;

    int c;
    int loadavg = 0;
    unsigned int i, j;

    struct rfc2045 *rfcbody = rfc2045_alloc();

    /* Remember the command line arguments */
    for(i=0;i<argc;i++){
	cmdsize += strlen(argv[i]) + 2;
    }
    cmdargs = malloc(cmdsize);
    for(i=0;i<argc;i++){
	strcat(cmdargs,argv[i]);
	if(i != argc-1) strcat(cmdargs," ");
    }
	
	/* Clear the rewrite_filename - this way, if it's non-NULL at cleanup, we'll know if we need to remove it */
	memset(rewrite_filename, 0, sizeof(rewrite_filename));

    /* Initialize */
    REMOVED = vlist_alloc();
    EXTENSIONS = vlist_alloc();
    ALLOW_USERS = vlist_alloc();
    RCPT_TOS = vlist_alloc();

    openlog(argv[0],LOG_PID,LOG_MAIL);	// log to mail

    /* Establish config */

    V_PROGNAME = strdup(argv[0]);	/* get program name */
    INTERACTIVE = isatty(fileno(stdout));    /* Find out if we are interactive */
    VIRGE_START_TIME = time(NULL);    /* Get the start time */

    /* Check/get the arguments/options */
    while ((c = getopt(argc, argv, "c:Cf:a:d:s:hD:O:RtS:k")) != EOF) {
	switch (c) {
	case 'k':
	    opt_keep++;
	    break;

	    /* Config file location */
	case 'c':
	    if (optarg[0] == '\0') {
		fprintf(stderr, "ERROR: '-c' used, but no configuration file specified - aborting!\n");
		exit(1);
	    }
	    if (!file_exists(optarg)) {
		fprintf(stderr, "ERROR: Configuration file '%s' does not exist - aborting!\n", optarg);
		exit(1);
	    }
	    V_CONFIG = strdup(optarg);
	    break;
			
	    /* Config check requested? */
	case 'C':
	    CHECK_CONFIG_VERBOSE = 1;
	    break;

	    /* Sender - as given by sendmail (-f <email>) */
	case 'f':
	    MAIL_FROM = strdup(optarg);
	    virge_debug(1, "[ main() ] MAIL_FROM set to: '%s'", MAIL_FROM);
	    break;

	    /* -a option for procmail */
	case 'a':
	    arg_a = strdup(optarg);
	    virge_debug(1, "[ main() ] arg_a set to: '%s'", arg_a);
	    break;
			
	    /* this will contain 1st 'from' header - used for local delivery */
	    /* This will be set if the -f was not specified by sendmail */
	case 's':
	    if (MAIL_FROM[0] == '\0') {
		MAIL_FROM = strdup(optarg);
		virge_debug(1,
			    "[ main() ] MAIL_FROM set to: '%s' using content from '-s', "
			    "because Sendmail didn't supply the sender using '-f'", MAIL_FROM);
	    }
	    break;

	    /* Recipient */
	case 'd':
	    vlist_append(RCPT_TOS, optarg);
	    virge_debug(1, "[ main() ] added RCPT_TO: '%s'", optarg);
	    break;
			
	    /* DEBUG = output ? */
	case 'O':
	    DEBUG_OUTPUT = atoi(optarg);
	    if(!DEBUG_OUTPUT) DEBUG_OUTPUT=3;		/* send to both */
	    CMDLINE_DEBUG = 1;

	    fprintf(stderr,"Debug level set to %d\n",DEBUG_OUTPUT);
	    break;

	    /* DEBUG_LEVEL = ? */
	case 'D':
	    DEBUG_LEVEL = atoi(optarg);
	    printf("debug level set to %d\n",DEBUG_LEVEL);
	    if(DEBUG_LEVEL==0){
		DEBUG_LEVEL = 1;		/* default */
	    }
	    CMDLINE_DEBUG_LEVEL = 1;
	    break;
			
	    /* Check regexp in extension file */
	case 'R':
	    CHECK_REGEX = 1;
	    break;

	case 'S':
	    DELIVER_PROCMAIL = 0;	/* mutually exclusive */
	    DELIVER_SMTP = 1;
	    SMTP_HOST    = strdup(optarg);
	    cc = strchr(SMTP_HOST,':');
	    if(cc){			/* is a port specified? */
		*cc = '\000';
		SMTP_PORT = atoi(cc+1);
	    }
	    virge_debug(0,"SMTP filter; will deliver to SMTP at %s:%d\n",SMTP_HOST,SMTP_PORT);
	    break;
		

	case 't':
	    V_TEST = 1;
	    break;

	    /* Help */
	default:
	case 'h':
	    usage();
	    break;
	}
    }

    syslog(LOG_INFO,"virge starting up");

    load_virge_config();		/* load configuration from file */


    virge_debug(1,"starting up. cmd='%s'",cmdargs);

    setup_signals();			/* set up signal handlers */
    check_current_config(); /* Check current Virge configuration (as loaded from a config file) */
    load_allowusers(); /* Load list of users for which mail should be delivered w/o checking */

    /* Make sure that we have a consistent set of delivery options
     */
    if(DELIVER_PROCMAIL + DELIVER_SMTP  != 1)
	{
		virge_debug(0, "DELIVER_PROCMAIL=%d DELIVER_SMTP=%d; only one can be set", DELIVER_PROCMAIL,DELIVER_SMTP);
		exit(1);
    }

    /* Do we need to test extensions for regular expression validity?
     * (Previously this was done with check_regex).
     * modified to use check_extension_regex().
     */
    if (CHECK_REGEX)
	{
		check_extension_regex(0);
		exit(0);
    }

    virge_drop_privs();		     /* Drop our privs if necessary */

    /* If no recepient has been specified and we are running as a PROCMAIL filter, then
     * we should definately isolate.
     */
    if (vlist_count(RCPT_TOS) == 0 && DELIVER_PROCMAIL && !V_TEST)
	{
		/* We will isolate this mail */
		ISOLATE = 1;

		/* Put "UNKNOWN" as the recipient */
		vlist_append(RCPT_TOS, "UNKNOWN");
		virge_debug(2, "[ main() ] No recipient specified (ouch!) - setting it to 'UNKNOWN' and performing isolation");
    }

    /* Create the messageid */
    virge_create_messageid();

    /* Create a name for temporary file */
    snprintf(input_filename, sizeof(input_filename), "%s/%s", V_TEMPDIR, "virge-XXXXXX");

    /* Open temporary file */
    if ((tempfile_fd = mkstemp(input_filename)) == -1)
	{
		virge_debug(2, "[ main() ] failed to mkstemp(\"%s\")", input_filename);
		syslog(LOG_INFO,"FAILURE: Cannot create temporary file '%s' with mkstemp() in main()",
		       input_filename);
		fprintf(stderr,"FAILURE: Cannot create temporary file '%s' with mkstemp() in main()",
			input_filename);
		exit(-1);
    }

    virge_debug(3, "[ main() ] tempfile created: '%s'", input_filename);

    input_file = fdopen(tempfile_fd,"w+"); /* where we are going to put our stuff */

    /* If we are doing procmail delivery, get the message on stdin into the temp file. */
    if(DELIVER_PROCMAIL)
		virge_read_message_stdin();

    /* If we are being an SMTP proxy, start up proxy and get message on stdin */
    if(DELIVER_SMTP)
	{
		if(smtp_open_proxy(input_file))
		{
		    /* got an error */
		    virge_defer(0);		/* defer */
		}
    }

    /* Make RCPT_TO. */
    make_rcpt_to();

    virge_debug(2, "RCPT_TO set to \"%s\"", RCPT_TO);

    /* Get email size. */
    if (fseek(input_file,0,SEEK_END) == -1 ||
		(V_MAIL_SIZE = ftell(input_file)) == -1 ||
		fseek(input_file,0,SEEK_SET) == -1)
    {
		event_log(LOG_ERROR, "Getting email size: %s", strerror(errno));
		virge_defer(0);
    }

    virge_debug(2, "MAIL_SIZE set to %u", V_MAIL_SIZE);
	
    /* Read the entire message into memory and rfc2045 parse it.
     * Ultimately, it would be better to build the buffer when we create the file,
     * or map it into memory with mmap.
     */
    message_buf = malloc(V_MAIL_SIZE+1);
    fseek(input_file,0,SEEK_SET);
    if(fread(message_buf,1,V_MAIL_SIZE,input_file)!=V_MAIL_SIZE){
	virge_debug(0,"[ main() ] failed to read() message!");
    }
    rfc2045_parse(rfcbody, message_buf, V_MAIL_SIZE);

    virge_debug(2, "rfc2045_parse done");

    /* Check if all the recipients are in ALLOW_USERS, and if so, delivery it immediately. */
    for (j = 0; j < vlist_count(RCPT_TOS); j ++)
    {
		for (i = 0; i < vlist_count(ALLOW_USERS) &&
			 strcmp(vlist(RCPT_TOS, j), vlist(ALLOW_USERS, i)); i ++);

		if (i == vlist_count(ALLOW_USERS))
		    break; /* Recipient not found in ALLOW_USERS. */
    }

    if (j == vlist_count(RCPT_TOS))
    {
		/* All recipients are in ALLOW_USERS. */
		virge_deliver(input_filename, rfcbody);
    }

    virge_debug(2, "Not all recipients are in ALLOW_USERS");

    /*
     * If recipient is missing, we will isolate the mail now - no need to decode attachments,
     * admin will (should :) probably check it anyway
     */
    if (ISOLATE != 0)
	{
		/* Isolate the mailfile */
		virge_debug(2, "[ main() ] isolating mail because no recipients were specified");

		/* No need to check return of isolate(), we'll defer the message anyway. */
		isolate(2, "No recipients have been specified.",
			input_filename, 1, "Mail has been isolated because no recipients have been specified.");
		virge_defer(rfcbody);
    }

    /* Check the load on the machine - if load exceeds the 'deliver_loadavg' param in config, deliver immediately */
    loadavg = check_loadavg();

    /* Deliver immediately if load is high */
    if (loadavg >= V_DELIVER_LOADAVG)	{
	virge_debug(1, "[ main() ] Direct delivery (from: '%s', to: '%s') because of high load on the machine",
		    MAIL_FROM, RCPT_TO);
	event_log(LOG_VIRGE, "Direct delivery [load too high] for mail from=<%s>, to=<%s>, size=<%lu bytes>",
		  MAIL_FROM, RCPT_TO, V_MAIL_SIZE);

	/* Delivery because load is high and we don't want to waste more resources */
	virge_deliver(input_filename,rfcbody);
    }

    if( loadavg >= V_DEFER_LOADAVG){
	virge_debug(1, "[ main() ] Load is too high; defering mail");
	virge_defer(rfcbody); // postfix_defer actually does exit.
    }

    /*
     * Get the headers of the mail (will need it later for sender notification)
     */
    get_headers(input_file);

    /* Decode the message... */
    rfc2045_decode(rfcbody, &do_print_structure, 0);

    virge_debug(2, "rfc2045_decode done");

    load_extensions();  /* Load the list of extensions/filenames to check */

    /* If there is just one section, then deliver normally */
    virge_debug(1, "[ main() ] sections=%d",sections);
    if(sections != 1)
	{
		int fd;

		snprintf(scratchdir, sizeof(scratchdir), "%s/XXXXXX", V_TEMPDIR);

		fd = mkstemp(scratchdir);
		if(fd<0)
		{
		    perror("no valid dir could be created");
		    virge_defer(rfcbody);
		}

		close(fd);
		unlink(scratchdir);		/* because I'm going to mkdir it */
	
		virge_debug(3, "[ main() ] scratch dir *name* created: '%s'", scratchdir);

		if (!virge_mkdir(scratchdir))
		{
		    /* Defer because scratchdir could not be delivered. */
		    virge_debug(2, "[ main() ] fatal error - scratch dir not created.");
		    virge_defer(rfcbody);
		}

		virge_debug(1, "[ main() ] scratch dir '%s' created", scratchdir);

		/* Create the scratchdir_prefix */
		strcpy(scratchdir_prefix,scratchdir);
		strcat(scratchdir_prefix,"/");

		/* Set umask  */
		umask(0777 & ~strtoul(VIRGE_DIR_MODE,NULL,8));
		virge_debug(3, "[ main() ] umask() set");

		/* Ok - now extract files there... */
		extract_section(rfcbody, NULL, scratchdir_prefix, 0, NULL, extract_file);
	
		virge_debug(1, "[ main() ] attachments extracted into '%s'", scratchdir);
	
		/* Scan for a virus */
		if(virge_vscan(rfcbody,scratchdir))
		{
		    event_log(LOG_ACTION, "virus=<%s> from=<%s> to=<%s> size=<%lu>",
			      VIRUS_NAME, MAIL_FROM, RCPT_TO, V_MAIL_SIZE);

		    if (isolate(0, "Virus has been found in the mail.", input_filename, 1, "Mail has been isolated because a virus has been found.") == -1)
			{
				/* Email is infected, but we failed isolating it or
				 * deliverying the notification emails, so defer. */
				virge_defer(rfcbody);
			}

		    /* Remove from RCPT_TOS all users that are not in ALLOW_USERS. */
		    keep_only_allowed_rcpt_tos();

		    /* If there's at least one user still in RCPT_TOS, deliver, otherwise, drop it. */
		    if (vlist_count(RCPT_TOS) > 0)
				virge_deliver(input_filename, rfcbody);

		    virge_cleanup(rfcbody,0);
		}

		virge_checkrewrite(rfcbody);		/* Scan for an illegal extension */
		virge_deliver(input_filename,rfcbody); /* If virge_rewrite() didn't deliver, we should */
    }

    /* One section, or we got an error.
     * There is more than 1 section - unpack & scan the mail
     */
    virge_deliver(input_filename,rfcbody);
    return(0);				/* silence gcc */
}

